package com.elitescloud.boot.tenant.client.support.impl;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.context.TenantSession;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.tenant.client.support.TenantProviderService;
import com.elitescloud.boot.wrapper.RedisWrapper;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.common.CloudtOptional;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitescloud.cloudt.tenant.provider.TenantProvider;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import lombok.extern.log4j.Log4j2;
import org.springframework.util.Assert;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2022/9/21
 */
@Log4j2
public class DefaultTenantProvider implements TenantProvider {

    private final TenantProviderService tenantProviderService;
    private final RedisUtils redisUtils;
    private final RedisWrapper redisWrapper;

    private final Cache<String, SysTenantDTO> tenantCache;

    private Cache<String, Boolean> queryFail = null;
    private static final String FAIL_KEY_ALL = "all";

    public DefaultTenantProvider(TenantProviderService tenantProviderService, RedisUtils redisUtils, RedisWrapper redisWrapper) {
        this.tenantProviderService = tenantProviderService;
        this.redisUtils = redisUtils;
        this.redisWrapper = redisWrapper;

        tenantCache = Caffeine.newBuilder()
                .maximumSize(400)
                .expireAfterWrite(Duration.ofMinutes(30))
                .build();
        queryFail = Caffeine.newBuilder()
                .maximumSize(20)
                .expireAfterWrite(Duration.ofMinutes(20))
                .build();
    }

    /**
     * 获取所有租户信息
     *
     * @return 租户信息
     */
    public Map<String, SysTenantDTO> getAllTenant() {
        var tenants = queryAllTenants();
        if (tenants.isEmpty()) {
            return Collections.emptyMap();
        }

        var tenantMap = tenants.stream()
                .collect(Collectors.toMap(t -> t.getId().toString(), t -> t, (t1, t2) -> t1));
        return Collections.unmodifiableMap(tenantMap);
    }

    /**
     * 根据租户ID获取租户信息
     *
     * @param id 租户ID
     * @return 租户信息
     */
    public CloudtOptional<SysTenantDTO> getById(Long id) {
        Assert.notNull(id, "租户ID为空");
        if (id.longValue() == TenantConstant.DEFAULT_TENANT_ID) {
            // 默认租户
            return CloudtOptional.empty();
        }

        SysTenantDTO tenant = tenantCache.getIfPresent("id_" + id);
        if (tenant != null) {
            return CloudtOptional.of(tenant);
        }

        var tenants = queryAllTenants();
        if (tenants.isEmpty()) {
            return CloudtOptional.empty();
        }

        tenant = tenants.stream()
                .filter(t -> t.getId().equals(id))
                .findFirst()
                .orElse(null);
        if (tenant != null) {
            tenantCache.put("id_" + id, tenant);
        }
        return CloudtOptional.of(tenant);
    }

    @Override
    public CloudtOptional<SysTenantDTO> getByCode(String code) {
        Assert.hasText(code, "租户编码为空");

        SysTenantDTO tenant = tenantCache.getIfPresent("code_" + code);
        if (tenant != null) {
            return CloudtOptional.of(tenant);
        }

        var tenants = queryAllTenants();
        if (tenants.isEmpty()) {
            return CloudtOptional.empty();
        }

        tenant = tenants.stream()
                .filter(t -> code.equals(t.getTenantCode()))
                .findFirst()
                .orElse(null);
        if (tenant != null) {
            tenantCache.put("code_" + code, tenant);
        }
        return CloudtOptional.of(tenant);
    }

    /**
     * 根据域名获取租户信息
     *
     * @param domain 域名
     * @return 租户信息
     */
    @Override
    public CloudtOptional<SysTenantDTO> getByDomain(String domain) {
        SysTenantDTO tenant = tenantCache.getIfPresent("domain_" + domain);
        if (tenant != null) {
            return CloudtOptional.of(tenant);
        }

        var tenants = queryAllTenants();
        if (tenants.isEmpty()) {
            return CloudtOptional.empty();
        }

        tenant = tenants.stream()
                .filter(t -> {
                    if (CharSequenceUtil.equals(t.getTenantDomain(), domain)) {
                        // 租户域名
                        return true;
                    }
                    return this.delimiterMatch(t.getCustomDomain(), domain);
                })
                .findFirst()
                .orElse(null);
        if (tenant != null) {
            tenantCache.put("domain_" + domain, tenant);
        }
        return CloudtOptional.of(tenant);
    }

    /**
     * 清理缓存
     */
    @Override
    public void clearCache() {
        tenantCache.invalidateAll();

        // 清理redis
        String cacheKey = TenantConstant.CACHE_ALL_KEY + ":all";
        redisWrapper.apply(() -> {
            redisUtils.del(cacheKey);
            return null;
        }, null);
    }

    protected List<SysTenantDTO> queryAllFromServer() {
        if (queryFail.getIfPresent(FAIL_KEY_ALL) != null) {
            // 最近有查询失败，则暂时不再查询
            return Collections.emptyList();
        }

        // 最后调用远程接口获取
        ApiResult<List<SysTenantDTO>> queryTenantResult = null;
        TenantSession.setNoTenant();
        try {
            queryTenantResult = tenantProviderService.allTenants();
        } catch (Exception e) {
            log.error("查询租户信息异常：", e);
            queryFail.put(FAIL_KEY_ALL, true);
            return Collections.emptyList();
        } finally {
            TenantSession.clearNoTenant();
        }
        if (!queryTenantResult.isSuccess() || queryTenantResult.getData() == null) {
            log.error("查询租户信息失败：{}", queryTenantResult.getMsg() + "；" + queryTenantResult.getErrorMsg());
            return Collections.emptyList();
        }

        if (queryTenantResult.getData().isEmpty()) {
            log.warn("未查询到有效租户信息");
        }
        return queryTenantResult.getData();
    }

    protected List<SysTenantDTO> queryAllFromRedis() {
        String cacheKey = TenantConstant.CACHE_ALL_KEY + ":all";

        Map<String, SysTenantDTO> redisAll = null;
        try {
            redisAll = (Map<String, SysTenantDTO>) redisWrapper.apply(() -> redisUtils.get(cacheKey), null);
        } catch (Exception e) {
            log.error("查询租户缓存失败：", e);
        }
        if (redisAll != null) {
            return new ArrayList<>(redisAll.values());
        }
        return Collections.emptyList();
    }

    private List<SysTenantDTO> queryAllTenants() {
        synchronized (FAIL_KEY_ALL) {
            // 判断缓存中是否已有
            List<SysTenantDTO> tenants = queryAllFromRedis();
            if (!tenants.isEmpty()) {
                return unmodified(tenants);
            }

            // 调用远程接口获取
            tenants = queryAllFromServer();
            if (tenants.isEmpty()) {
                // 避免频繁查询
                queryFail.put(FAIL_KEY_ALL, true);
                return Collections.emptyList();
            }

            // 缓存租户信息
            cacheToRedis(tenants);

            return unmodified(tenants);
        }
    }

    private void cacheToRedis(List<SysTenantDTO> tenants) {
        String cacheKey = TenantConstant.CACHE_ALL_KEY + ":all";

        var tenantMap = tenants.stream()
                .collect(Collectors.toMap(t -> t.getId().toString(), t -> t, (t1, t2) -> t1));
        try {
            redisWrapper.apply(() -> {
                redisUtils.set(cacheKey, tenantMap);
                return null;
            }, null);
        } catch (Exception e) {
            log.error("设置租户缓存失败：", e);
        }
    }

    private boolean delimiterMatch(String delimiterStr, String matcherStr) {
        if (CharSequenceUtil.isBlank(delimiterStr)) {
            return false;
        }
        for (String s : delimiterStr.split(",")) {
            if (CharSequenceUtil.equals(s, matcherStr)) {
                return true;
            }
        }
        return false;
    }

    private List<SysTenantDTO> unmodified(List<SysTenantDTO> tenants) {
        if (tenants.isEmpty()) {
            return Collections.emptyList();
        }
        for (SysTenantDTO tenant : tenants) {
            if (tenant.getAppCodes() == null) {
                tenant.setAppCodes(Collections.emptySet());
            } else {
                tenant.setAppCodes(Collections.unmodifiableSet(tenant.getAppCodes()));
            }
        }
        return Collections.unmodifiableList(tenants);
    }
}
