package com.elitescloud.boot.webservice.support;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.cloudt.common.base.annotation.webservice.WebServiceConsumer;
import com.elitescloud.boot.webservice.WebServiceProperties;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.annotation.AnnotatedBeanDefinition;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.*;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.ClassPathScanningCandidateComponentProvider;
import org.springframework.core.env.Environment;
import org.springframework.core.type.filter.AnnotationTypeFilter;
import org.springframework.lang.NonNull;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.HashSet;
import java.util.Set;

/**
 * 服务消费者配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/8/11
 */
@Log4j2
public abstract class AbstractServiceConsumerConfig implements ServiceConsumerConfig, EnvironmentAware, BeanDefinitionRegistryPostProcessor {

    protected ConfigurableListableBeanFactory beanFactory;
    protected Environment environment;
    private WebServiceProperties properties;

    @Override
    public void setEnvironment(@NonNull Environment environment) {
        this.environment = environment;
    }

    @Override
    public void postProcessBeanFactory(@NonNull ConfigurableListableBeanFactory beanFactory) throws BeansException {
        this.beanFactory = beanFactory;
    }

    @Override
    public void postProcessBeanDefinitionRegistry(@NonNull BeanDefinitionRegistry registry) throws BeansException {
        Set<String> consumerPackages = getProperties().getConsumerPackages();
        if (CollectionUtils.isEmpty(consumerPackages)) {
            // 没有配置客户端
            return;
        }

        // 扫描消费者接口
        Set<BeanDefinition> definitions = scanConsumer(consumerPackages);

        // 开始注册bean
        for (BeanDefinition definition : definitions) {
            Class<?> serviceClass = null;
            try {
                serviceClass = Class.forName(definition.getBeanClassName());
            } catch (ClassNotFoundException e) {
                throw new RuntimeException(e);
            }
            var consumer = serviceClass.getAnnotation(WebServiceConsumer.class);
            String beanName = generateBeanName(definition, consumer);

            // 注册bean
            var bean = buildConsumer(serviceClass, beanName, consumer);
            registry.registerBeanDefinition(beanName, buildBeanDefinition(serviceClass, bean));
            log.info("注册WebService客户端Bean：{}, 服务接口：{}", beanName, serviceClass.getName());
        }
    }

    /**
     * 获取配置
     *
     * @return 配置
     */
    protected WebServiceProperties getProperties() {
        if (properties == null) {
            var bindResult = Binder.get(environment).bind(WebServiceProperties.CONFIG_PREFIX, WebServiceProperties.class);
            if (bindResult.isBound()) {
                properties = bindResult.get();
            }
        }

        return properties;
    }

    /**
     * 获取配置的地址
     *
     * @param addressKey 地址key
     * @return 地址
     */
    protected String getConfigAddress(String addressKey) {
        for (WebServiceProperties.ConsumerAddress consumerAddress : getProperties().getConsumerAddresses()) {
            if (CharSequenceUtil.equals(addressKey, consumerAddress.getAddressKey())) {
                return consumerAddress.getAddress();
            }
        }

        return null;
    }

    private BeanDefinition buildBeanDefinition(Class<?> serviceClass, Object bean) {
        GenericBeanDefinition beanDefinition = (GenericBeanDefinition) BeanDefinitionBuilder.genericBeanDefinition(serviceClass).getRawBeanDefinition();
        beanDefinition.setBeanClass(ServiceConsumerFactory.class);
        beanDefinition.getConstructorArgumentValues().addGenericArgumentValue(serviceClass);
        beanDefinition.getConstructorArgumentValues().addGenericArgumentValue(bean);

        beanDefinition.setAutowireMode(AbstractBeanDefinition.AUTOWIRE_BY_TYPE);
        return beanDefinition;
    }

    private String generateBeanName(BeanDefinition beanDefinition, WebServiceConsumer consumer) {
        if (StringUtils.hasText(consumer.beanName())) {
            return consumer.beanName();
        }

        return beanDefinition.getBeanClassName() + "WsProxy";
    }

    /**
     * 扫描消费者
     *
     * @param basePackages 包路径
     * @return 消费者
     */
    private Set<BeanDefinition> scanConsumer(Set<String> basePackages) {
        Set<BeanDefinition> consumers = new HashSet<>();
        var componentProvider = new CustomClassPathScanningCandidateComponentProvider();
        componentProvider.addIncludeFilter(new AnnotationTypeFilter(WebServiceConsumer.class));

        for (String basePackage : basePackages) {
            var beanDefinitions = componentProvider.findCandidateComponents(basePackage.replace(".", "/"));
            consumers.addAll(beanDefinitions);
        }

        return consumers;
    }

    static class CustomClassPathScanningCandidateComponentProvider extends ClassPathScanningCandidateComponentProvider {
        public CustomClassPathScanningCandidateComponentProvider() {
            super(false);
        }

        @Override
        protected boolean isCandidateComponent(AnnotatedBeanDefinition beanDefinition) {
            return true;
        }
    }

    static class ServiceConsumerFactory<T> implements FactoryBean<T> {

        private final Class<T> serviceType;
        private final Object proxy;

        public ServiceConsumerFactory(Class<T> serviceType, Object proxy) {
            this.serviceType = serviceType;
            this.proxy = proxy;
        }

        @SuppressWarnings("unchecked")
        @Override
        public T getObject() throws Exception {
            return (T) proxy;
        }

        @Override
        public Class<?> getObjectType() {
            return serviceType;
        }
    }
}
