package com.elitescloud.boot.websocket;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.websocket.handler.WebSocketMsgPublishable;
import com.elitescloud.boot.websocket.support.WebSocketSessionManager;
import com.elitescloud.boot.websocket.util.WebSocketUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.socket.TextMessage;
import org.springframework.web.socket.WebSocketSession;

import javax.validation.constraints.NotNull;
import java.io.IOException;
import java.io.Serializable;
import java.util.List;

/**
 * WebSocket工具.
 *
 * @author Kaiser（wang shao）
 * @date 2023/5/31
 */
@Slf4j
public class CloudtWebSocketTemplate {

    private final WebSocketMsgPublishable msgPublishable;

    public CloudtWebSocketTemplate(WebSocketMsgPublishable msgPublishable) {
        this.msgPublishable = msgPublishable;
    }

    /**
     * 发送消息
     *
     * @param result    消息
     * @param usernames 接收账号
     * @param <T>       消息类型
     */
    public <T extends Serializable> void sendMsg(@NotNull ApiResult<T> result, String... usernames) {
        var msg = WebSocketUtil.convertTextMessage(result);
        for (String username : usernames) {
            var session = WebSocketSessionManager.getSession(username);
            if (CollUtil.isNotEmpty(session)) {
                sendTxtMsg(msg, username, session);
            } else if (msgPublishable != null) {
                // 当前服务未找到，需要分发，由其它服务分发
                msgPublishable.publish(msg.getPayload(), username);
            }
        }
    }

    private void sendTxtMsg(TextMessage msg, String username, List<WebSocketSession> sessions) {
        boolean sent = false;
        for (WebSocketSession session : sessions) {
            if (!session.isOpen()) {
                // session已关闭
                WebSocketSessionManager.removeSession(session);
                continue;
            }

            try {
                session.sendMessage(msg);
                sent = true;
            } catch (IOException e) {
                log.error("发送消息[{}]异常：", msg, e);
            }
        }

        if (!sent && msgPublishable != null) {
            // 未发送成功
            msgPublishable.publish(msg.getPayload(), username);
        }
    }
}
