package com.elitescloud.cloudt.context.util;

import org.springframework.core.io.UrlResource;
import org.springframework.core.io.support.PropertiesLoaderUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Nullable;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.io.IOException;
import java.net.URL;
import java.util.*;

/**
 * Properties工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2023/1/9
 */
public class PropertiesUtil {
    private PropertiesUtil() {
    }

    /**
     * 加载所有的配置
     *
     * @param resourceName 资源文件名称
     * @param classLoader  classLoader
     * @return 配置
     */
    public static Map<String, List<String>> loadResourceProperties(@NotBlank String resourceName, @Nullable ClassLoader classLoader) {
        // 加载配置文件
        var resources = ResourceUtil.loadResource(resourceName, classLoader);
        if (resources.isEmpty()) {
            return Collections.emptyMap();
        }

        return loadProperties(resources, true);
    }

    /**
     * 加载所有的配置
     *
     * @param urls  配置文件url
     * @param merge 是否合并，true：合并结果，false：后者覆盖前者
     * @return 配置
     */
    public static Map<String, List<String>> loadProperties(@NotEmpty List<URL> urls, boolean merge) {
        if (urls.isEmpty()) {
            return Collections.emptyMap();
        }
        Map<String, List<String>> result = new HashMap<>(128);
        for (URL url : urls) {
            var properties = loadProperties(url);
            if (merge) {
                // 合并属性
                for (Map.Entry<String, List<String>> entry : properties.entrySet()) {
                    if (!result.containsKey(entry.getKey())) {
                        result.put(entry.getKey(), entry.getValue());
                        continue;
                    }
                    var valueList = result.get(entry.getKey());
                    for (String value : entry.getValue()) {
                        if (valueList.contains(value)) {
                            continue;
                        }
                        valueList.add(value);
                    }
                }
            } else {
                result.putAll(properties);
            }
        }
        return result;
    }

    /**
     * 加载配置
     *
     * @param url 配置文件的url
     * @return 配置
     */
    public static Map<String, List<String>> loadProperties(@NotNull URL url) {
        if (url == null) {
            return Collections.emptyMap();
        }
        Map<String, List<String>> result = new HashMap<>(128);
        UrlResource resource = new UrlResource(url);
        Properties properties = null;
        try {
            properties = PropertiesLoaderUtils.loadProperties(resource);
        } catch (IOException e) {
            throw new IllegalArgumentException("加载配置" + url + "异常", e);
        }
        for (Map.Entry<?, ?> entry : properties.entrySet()) {
            String factoryTypeName = ((String) entry.getKey()).trim();
            String[] factoryImplementationNames =
                    StringUtils.commaDelimitedListToStringArray((String) entry.getValue());
            for (String factoryImplementationName : factoryImplementationNames) {
                result.computeIfAbsent(factoryTypeName, key -> new ArrayList<>())
                        .add(factoryImplementationName.trim());
            }
        }

        return result;
    }
}
