package com.elitescloud.boot.util;

import cn.hutool.core.text.CharSequenceUtil;
import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.jwk.JWKSet;
import com.nimbusds.jose.jwk.RSAKey;
import com.nimbusds.jose.jwk.source.JWKSource;
import com.nimbusds.jose.proc.SecurityContext;
import com.nimbusds.jwt.EncryptedJWT;
import com.nimbusds.jwt.PlainJWT;
import com.nimbusds.jwt.SignedJWT;
import lombok.extern.log4j.Log4j2;
import org.springframework.core.io.Resource;
import org.springframework.lang.NonNull;
import org.springframework.security.oauth2.jwt.JwtDecoder;
import org.springframework.security.oauth2.jwt.JwtEncoder;
import org.springframework.security.oauth2.jwt.NimbusJwtDecoder;
import org.springframework.security.oauth2.jwt.NimbusJwtEncoder;
import org.springframework.util.Assert;

import java.io.IOException;
import java.io.InputStream;
import java.security.KeyStore;
import java.text.ParseException;
import java.util.Collections;
import java.util.Map;

/**
 * Jwt工具类.
 * <p>
 * 基于 <a href = "https://bitbucket.org/connect2id/nimbus-jose-jwt/src/master">Nimbus JOSE + JWT</a> 实现的工具类
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/1
 */
@Log4j2
public class JwtUtil {

    private JwtUtil() {
    }

    /**
     * 构建jwt解密器
     *
     * @param rsaKey rsaKey
     * @return decoder
     */
    public static JwtDecoder buildJwtDecoder(@NonNull RSAKey rsaKey) {
        Assert.notNull(rsaKey, "rsaKey为空");

        try {
            return NimbusJwtDecoder.withPublicKey(rsaKey.toRSAPublicKey()).build();
        } catch (JOSEException e) {
            log.error("jwtDecoder构建失败：", e);
            throw new IllegalArgumentException(e);
        }
    }

    /**
     * 构建jwt加密器
     *
     * @param rsaKey rsaKey
     * @return decoder
     */
    public static JwtEncoder buildJwtEncoder(@NonNull RSAKey rsaKey) {
        Assert.notNull(rsaKey, "rsaKey为空");

        JWKSource<SecurityContext> jwkSource = generateJwkSource(rsaKey);
        return new NimbusJwtEncoder(jwkSource);
    }

    /**
     * 加载RSAKey
     *
     * @param keyStore keyStore
     * @param alias    key别名
     * @param secret   key Secret
     * @return RSAKey
     */
    public static RSAKey loadRSAKey(@NonNull KeyStore keyStore, String alias, @NonNull String secret) {
        Assert.notNull(keyStore, "keyStore为空");
        Assert.hasText(alias, "alias为空");
        Assert.hasText(secret, "secret为空");

        // 加载RSAKey
        try {
            return RSAKey.load(keyStore, alias, secret.toCharArray());
        } catch (Exception e) {
            log.error("rsaKey加载失败：", e);
            throw new IllegalArgumentException(e);
        }
    }

    /**
     * 加载KeyStore工具
     *
     * @param keystoreStream keyStore流
     * @param type           keyStore类型
     * @param password       keystore密码
     * @param alias          key别名
     * @param secret         key secret
     * @return KeyStore
     */
    public static KeyStore loadKeystore(@NonNull InputStream keystoreStream, String type, String password, String alias, String secret) {
        Assert.hasText(password, "password为空");
        Assert.hasText(alias, "alias为空");
        Assert.hasText(secret, "secret为空");

        KeyStore keyStore = null;
        try {
            keyStore = KeyStore.getInstance(type);
            keyStore.load(keystoreStream, password.toCharArray());
        } catch (Exception e) {
            log.error("keystore加载失败：", e);
            throw new IllegalArgumentException(e);
        }

        return keyStore;
    }

    /**
     * 加载KeyStore工具
     *
     * @param keystoreResource keyStore资源
     * @param type             keyStore类型
     * @param password         keystore密码
     * @param alias            key别名
     * @param secret           key secret
     * @return KeyStore
     */
    public static KeyStore loadKeystore(@NonNull Resource keystoreResource, String type, @NonNull String password, String alias, String secret) {
        Assert.isTrue(keystoreResource.exists(), "keystore不存在");

        InputStream keystoreStream = null;
        try {
            keystoreStream = keystoreResource.getInputStream();
        } catch (IOException e) {
            log.error("加载keystore失败：", e);
            throw new IllegalArgumentException("加载keystore失败", e);
        }

        return loadKeystore(keystoreStream, type, password, alias, secret);
    }

    /**
     * 生成JWKSource
     *
     * @param rsaKey rsaKey
     * @return jwkSource
     */
    public static JWKSource<SecurityContext> generateJwkSource(@NonNull RSAKey rsaKey) {
        JWKSet jwkSet = new JWKSet(rsaKey);
        return (jwkSelector, securityContext) -> jwkSelector.select(jwkSet);
    }

    /**
     * 通用解密
     *
     * @param jwtStr jwt字符串
     * @return 解密结果
     */
    public static Map<String, Object> decode(String jwtStr) {
        if (CharSequenceUtil.isBlank(jwtStr)) {
            return Collections.emptyMap();
        }

        var partSize = jwtStr.split("\\.").length;
        try {
            if (partSize == 3) {
                return SignedJWT.parse(jwtStr).getPayload().toJSONObject();
            } else if (partSize == 2) {
                return PlainJWT.parse(jwtStr).getPayload().toJSONObject();
            } else if (partSize == 5) {
                return EncryptedJWT.parse(jwtStr).getPayload().toJSONObject();
            }
        } catch (ParseException e) {
            log.error("解密jwt失败：{}", jwtStr);
            throw new IllegalArgumentException("解密jwt失败：", e);
        }

        throw new IllegalStateException("暂不支持的jwt格式");
    }
}
