package com.elitescloud.boot.util;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotNull;
import java.util.Collection;
import java.util.function.Consumer;
import java.util.function.Supplier;

/**
 * 对象工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2023/9/28
 */
public class ObjUtil extends ObjectUtil {
    private ObjUtil() {
    }

    /**
     * 如果值不为空时消费
     *
     * @param value    值
     * @param consumer 消费者
     */
    public static <T> void ifNotNull(T value, @NotNull Consumer<T> consumer) {
        if (consumer == null) {
            return;
        }
        if (value != null) {
            consumer.accept(value);
        }
    }

    /**
     * 如果值为空时消费默认值
     *
     * @param value        值
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static <T> void ifNull(T value, T defaultValue, @NotNull Consumer<T> consumer) {
        ifNull(value, (Supplier<T>) (() -> defaultValue), consumer);
    }

    /**
     * 如果值为空时消费默认值
     *
     * @param value        值
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static <T> void ifNull(T value, Supplier<T> defaultValue, @NotNull Consumer<T> consumer) {
        if (consumer == null) {
            return;
        }
        if (value == null) {
            consumer.accept(defaultValue.get());
        }
    }

    /**
     * 如果值不为空则消费值
     *
     * @param value    待消费的值
     * @param consumer 消费者
     */
    public static void ifNotBlank(String value, @NotNull Consumer<String> consumer) {
        if (consumer == null) {
            return;
        }
        if (StringUtils.hasText(value)) {
            consumer.accept(value);
        }
    }

    /**
     * 如果值为空则消费默认值
     *
     * @param value        待消费的值
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static void ifBlank(String value, String defaultValue, @NotNull Consumer<String> consumer) {
        ifBlank(value, () -> defaultValue, consumer);
    }

    /**
     * 如果值为空则消费默认值
     *
     * @param value        待消费的值
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static void ifBlank(String value, Supplier<String> defaultValue, @NotNull Consumer<String> consumer) {
        if (consumer == null) {
            return;
        }
        if (!StringUtils.hasText(value)) {
            consumer.accept(defaultValue.get());
        }
    }

    /**
     * 如果集合不为空时消费
     *
     * @param value    集合
     * @param consumer 消费者
     */
    public static <T> void ifNotEmpty(Collection<T> value, @NotNull Consumer<Collection<T>> consumer) {
        if (consumer == null) {
            return;
        }
        if (CollUtil.isNotEmpty(value)) {
            consumer.accept(value);
        }
    }

    /**
     * 如果集合为空时消费默认值
     *
     * @param value        集合
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static <T> void ifEmpty(Collection<T> value, Collection<T> defaultValue, @NotNull Consumer<Collection<T>> consumer) {
        ifEmpty(value, () -> defaultValue, consumer);
    }

    /**
     * 如果集合为空时消费默认值
     *
     * @param value        集合
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static <T> void ifEmpty(Collection<T> value, Supplier<Collection<T>> defaultValue, @NotNull Consumer<Collection<T>> consumer) {
        if (consumer == null) {
            return;
        }
        if (CollUtil.isEmpty(value)) {
            consumer.accept(defaultValue.get());
        }
    }

    /**
     * 如果数组不为空时消费
     *
     * @param value    数组
     * @param consumer 消费者
     */
    public static <T> void ifNotEmpty(T[] value, @NotNull Consumer<T[]> consumer) {
        if (consumer == null) {
            return;
        }
        if (ArrayUtil.isNotEmpty(value)) {
            consumer.accept(value);
        }
    }

    /**
     * 如果数组为空时消费默认值
     *
     * @param value        数组
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static <T> void ifEmpty(T[] value, T[] defaultValue, @NotNull Consumer<T[]> consumer) {
        ifEmpty(value, () -> defaultValue, consumer);
    }

    /**
     * 如果集合为空时消费默认值
     *
     * @param value        集合
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static <T> void ifEmpty(T[] value, Supplier<T[]> defaultValue, @NotNull Consumer<T[]> consumer) {
        if (consumer == null) {
            return;
        }
        if (ArrayUtil.isEmpty(value)) {
            consumer.accept(defaultValue.get());
        }
    }
}
