package com.elitescloud.cloudt.common.base;

import com.elitescloud.boot.exception.BusinessException;

import javax.validation.constraints.NotNull;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;

/**
 * REST API 返回结果
 *
 * @author Michael Li
 * @date 2018-11-08
 */
public class ApiResult<T> extends BaseResult<T, ApiResult<T>> {

    private static final long serialVersionUID = 7722914707623525357L;
    private static BiFunction<Integer, String, String> errorMsgConvert = (code, errorNo) -> "系统出现异常,错误号:" + errorNo + ", 请联系管理员";

    /**
     * 错误详细信息
     */
    private String errorMsg;
    /**
     * 错误来源
     */
    private ErrorSource source;
    /**
     * 显示类型
     */
    private ErrorDisplayType displayType;
    /**
     * 详细错误信息
     */
    private ErrorDetail detail;
    /**
     * 链路跟踪ID
     */
    private String traceId;

    public ApiResult() {
        super.setTime(new Date());
    }

    public String getErrorMsg() {
        return errorMsg;
    }

    public ApiResult<T> setErrorMsg(String errorMsg) {
        this.errorMsg = errorMsg;
        return this;
    }

    public ApiResult<T> errorMsg(String errorMsg) {
        return setErrorMsg(errorMsg);
    }

    public ErrorSource getSource() {
        return source;
    }

    public ApiResult<T> setSource(ErrorSource source) {
        this.source = source;
        return this;
    }

    public ApiResult<T> source(ErrorSource source) {
        this.source = source;
        return this;
    }

    public ErrorDisplayType getDisplayType() {
        return displayType;
    }

    public ApiResult<T> setDisplayType(ErrorDisplayType displayType) {
        this.displayType = displayType;
        return this;
    }

    public ApiResult<T> displayType(ErrorDisplayType displayType) {
        this.displayType = displayType;
        return this;
    }

    public ApiResult<T> setDetail(ErrorDetail detail) {
        this.detail = detail;
        return this;
    }

    public ApiResult<T> detail(ErrorDetail detail) {
        this.detail = detail;
        return this;
    }

    public ErrorDetail getDetail() {
        return detail;
    }

    public String getTraceId() {
        return traceId;
    }

    public ApiResult<T> setTraceId(String traceId) {
        this.traceId = traceId;
        return this;
    }

    public ApiResult<T> traceId(String traceId) {
        this.traceId = traceId;
        return this;
    }

    /**
     * 是否失败
     *
     * @return 是否失败
     */
    public boolean isFailed() {
        return !super.isSuccess();
    }

    /**
     * 构建器
     *
     * @param <T> 数据类型
     * @return result
     */
    public static <T> ApiResult<T> builder() {
        return new ApiResult<>();
    }

    /**
     * 返回构造器结果
     *
     * @return result
     */
    public ApiResult<T> build() {
        return this;
    }

    /**
     * 转RpcResult
     *
     * @return rpcResult
     */
    public RpcResult<T> toRpcResult() {
        return new RpcResult<T>()
                .code(getCode())
                .msg(getMsg())
                .success(isSuccess())
                .errorNo(getErrorNo())
                .data(getData())
                .time(getTime());

    }

    /**
     * 安全获取数据
     *
     * @return 数据
     */
    public T computeData() {
        if (this.isFailed()) {
            String msg = getMsg();
            if (msg == null) {
                msg = getErrorMsg();
            }
            throw new BusinessException(msg);
        }
        return getData();
    }

    /**
     * 转换数据
     *
     * @param mapper 转换器
     * @param <R>    新数据类型
     * @return 转换结果
     */
    public <R> ApiResult<R> map(@NotNull Function<T, R> mapper) {
        ApiResult<R> result = new ApiResult<>();
        result.setErrorMsg(getErrorMsg());
        result.setTraceId(getTraceId());
        result.setCode(getCode());
        result.setMsg(getMsg());
        result.setSuccess(isSuccess());
        result.setErrorNo(getErrorNo());
        result.setData(mapper.apply(getData()));
        result.setTime(getTime());

        return result;
    }

    /**
     * 当成功时处理数据
     *
     * @param consumer 消费者
     * @return 结果
     */
    public ApiResult<T> peek(Consumer<T> consumer) {
        if (consumer == null) {
            return this;
        }
        if (this.isSuccess()) {
            consumer.accept(getData());
        }
        return this;
    }

    /**
     * 返回结果
     *
     * @param suc 是否成功
     * @param <T> 数据类型
     * @return result
     */
    public static <T> ApiResult<T> result(boolean suc) {
        if (suc) {
            return ok();
        }
        return fail();
    }

    /**
     * 返回结果
     *
     * @param apiCode ApiCode
     * @param <T>     数据类型
     * @return result
     */
    public static <T> ApiResult<T> result(ApiCode apiCode) {
        return result(apiCode, null);
    }

    /**
     * 返回结果
     *
     * @param apiCode ApiCode
     * @param data    数据
     * @param <T>     数据类型
     * @return result
     */
    public static <T> ApiResult<T> result(ApiCode apiCode, T data) {
        return result(apiCode, null, data);
    }

    /**
     * 返回结果
     *
     * @param apiCode ApiCode
     * @param msg     返回信息
     * @param data    数据
     * @param <T>     数据类型
     * @return result
     */
    public static <T> ApiResult<T> result(ApiCode apiCode, String msg, T data) {
        return result(apiCode, null, msg, null, data);
    }

    /**
     * 返回结果
     *
     * @param apiCode  ApiCode
     * @param errorNo  错误号
     * @param msg      信息提示
     * @param errorMsg 错误详细信息
     * @param data     数据
     * @param <T>      数据类型
     * @return result
     */
    public static <T> ApiResult<T> result(ApiCode apiCode, String errorNo, String msg, String errorMsg, T data) {
        var success = apiCode == ApiCode.SUCCESS;
        // 友好提示
        var msgFriendly = isBlank(msg) ? apiCode.getMsgFriendly() : msg;
        if (isBlank(msgFriendly)) {
            msgFriendly = errorMsgConvert.apply(apiCode.getCode(), errorNo);
        }

        return ApiResult.<T>builder()
                .code(apiCode.getCode())
                .errorNo(errorNo)
                .msg(msgFriendly)
                .errorMsg(errorMsg)
                .success(success)
                .data(data)
                .build();
    }

    /**
     * 返回结果
     *
     * @param code    错误码
     * @param errorNo 错误号
     * @param msg     信息提示
     * @param data    数据
     * @param <T>     数据类型
     * @return result
     */
    public static <T> ApiResult<T> result(int code, String errorNo, String msg, String errorMsg, T data) {
        var success = code >= 200 && code < 300;
        var msgFriendly = isBlank(msg) ? errorMsgConvert.apply(code, errorNo) : msg;

        return ApiResult.<T>builder()
                .code(code)
                .errorNo(errorNo)
                .msg(msgFriendly)
                .errorMsg(errorMsg)
                .data(data)
                .success(success)
                .build();
    }

    public static <T> ApiResult<T> ok() {
        return ok(null);
    }

    public static <T> ApiResult<T> ok(T data) {
        return result(ApiCode.SUCCESS, data);
    }

    public static <T> ApiResult<T> ok(T data, String msg) {
        return result(ApiCode.SUCCESS, msg, data);
    }

    public static <T> ApiResult<HashMap<String, T>> okMap(String key, T value) {
        HashMap<String, T> map = new HashMap<>(4);
        map.put(key, value);
        return ok(map);
    }

    public static Map<String, Object> toMap(ApiResult<?> result) {
        HashMap<String, Object> map = new HashMap<>(8);
        map.put("code", result.getCode());
        map.put("msg", result.getMsg());
        map.put("data", result.getData());
        map.put("success", result.isSuccess());
        map.put("time", result.getTime());

        return map;
    }

    public static <T> ApiResult<T> fail(ApiCode apiCode) {
        return fail(apiCode, null);
    }

    public static <T> ApiResult<T> fail(String msg) {
        return result(ApiCode.FAIL, msg, null);
    }

    public static <T> ApiResult<T> fail(int code, String msg) {
        return result(code, null, msg, null, null);
    }

    public static <T> ApiResult<T> fail(int code, String errorNo, T data, String msg) {
        return fail(code, errorNo, null, data, msg);
    }

    public static <T> ApiResult<T> fail(int code, String errorNo, String errorMsg, T data, String msg) {
        return result(code, errorNo, msg, errorMsg, data);
    }

    public static <T> ApiResult<T> fail(ApiCode apiCode, String msg) {
        if (ApiCode.SUCCESS == apiCode) {
            throw new IllegalArgumentException("失败结果状态码不能为" + ApiCode.SUCCESS.getCode());
        }
        return result(apiCode, msg, null);
    }

    public static <T> ApiResult<T> fail(ApiCode apiCode, T data) {
        if (ApiCode.SUCCESS == apiCode) {
            throw new IllegalArgumentException("失败结果状态码不能为" + ApiCode.SUCCESS.getCode());
        }
        return result(apiCode, data);
    }

    public static <T> ApiResult<T> fail(ApiCode apiCode, T data, String msg) {
        if (ApiCode.SUCCESS == apiCode) {
            throw new IllegalArgumentException("失败结果状态码不能为" + ApiCode.SUCCESS.getCode());
        }
        return result(apiCode, msg, data);
    }

    public static <T> ApiResult<T> fail(ApiCode apiCode, String errorNo, String errorMsg, T data, String msg) {
        if (ApiCode.SUCCESS == apiCode) {
            throw new IllegalArgumentException("失败结果状态码不能为" + ApiCode.SUCCESS.getCode());
        }
        return result(apiCode, errorNo, msg, errorMsg, data);
    }

    public static <T> ApiResult<HashMap<String, T>> fail(String key, T value) {
        HashMap<String, T> map = new HashMap<>(1);
        map.put(key, value);
        return result(ApiCode.FAIL, map);
    }

    public static <T> ApiResult<T> fail() {
        return fail(ApiCode.FAIL);
    }

    public static <T> ApiResult<T> noData() {
        return fail(ApiCode.NOT_FOUND_DATA);
    }

    public static void setErrorMsgConvert(BiFunction<Integer, String, String> errorMsgConvert) {
        ApiResult.errorMsgConvert = errorMsgConvert;
    }

    private static boolean isBlank(String str) {
        return str == null || str.trim().length() == 0;
    }

    @Override
    public String toString() {
        return "ApiResult{" +
                "errorMsg='" + errorMsg + '\'' +
                ", baseResult='" + super.toString() + '\'' +
                '}';
    }
}