package com.elitescloud.cloudt.common.base.common;

import javax.annotation.Nonnull;
import java.io.Serializable;
import java.util.*;

/**
 * 安全的枚举类.
 * <p>
 * 用以解决枚举类在不同系统之间传输时的不安全性
 *
 * @author Kaiser（wang shao）
 * 2022/9/21
 */
public class SafeEnum<T extends SafeEnum<T>> implements Serializable, Comparable<T> {
    private static final long serialVersionUID = 3946513070116495683L;

    private final String value;
    private final String description;

    private static final Map<Class<?>, TreeSet> INSTANCES = new HashMap<>(128);

    public SafeEnum() {
        this.value = null;
        this.description = null;
    }

    public SafeEnum(String value) {
        this.value = value;

        if (value != null) {
            var obj = valueOf(getClass(), value);
            this.description = obj == null ? null : obj.getDescription();
        } else {
            this.description = null;
        }
    }

    @SuppressWarnings("unchecked")
    protected SafeEnum(String value, String description) {
        this.value = value;
        this.description = description;
        if (value == null) {
            return;
        }
        INSTANCES.computeIfAbsent(getClass(), t -> new TreeSet<>()).add(this);
    }

    public String getValue() {
        return value;
    }

    public String getDescription() {
        return description;
    }

    @SuppressWarnings("unchecked")
    protected static <T> Set<T> all(Class<T> clazz) {
        var res = INSTANCES.get(clazz);
        if (res == null) {
            return Collections.emptySet();
        }
        return (Set<T>) Collections.unmodifiableSet(res);
    }

    @SuppressWarnings("unchecked")
    protected static <T> T valueOf(Class<T> clazz, String v) {
        if (v == null) {
            return null;
        }
        var res = INSTANCES.get(clazz);
        if (res == null) {
            return null;
        }
        for (Object r : res) {
            SafeEnum<?> e = (SafeEnum<?>) r;
            if (Objects.equals(e.getValue(), v)) {
                return (T) r;
            }
        }
        return null;
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(getValue());
    }

    @Override
    public boolean equals(Object obj) {
        if (obj instanceof SafeEnum) {
            SafeEnum<?> another = (SafeEnum<?>) obj;
            return Objects.equals(getValue(), another.getValue());
        }
        return false;
    }

    @Override
    public int compareTo(@Nonnull T o) {
        return Objects.compare(this, o, Comparator.comparing(SafeEnum::getValue));
    }

    @Override
    public String toString() {
        return getValue();
    }
}
