package com.elitescloud.cloudt.platform.controller.api;

import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.CodeNameParam;
import com.elitescloud.cloudt.constant.SysNumType;
import com.elitescloud.cloudt.platform.model.params.seq.SysPlatformNumberRuleAddParam;
import com.elitescloud.cloudt.platform.model.params.seq.SysPlatformNumberRuleQParam;
import com.elitescloud.cloudt.platform.model.params.seq.SysPlatformNumberRuleVO;
import com.elitescloud.cloudt.platform.service.ISysPlatformNumberRuleService;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 发号器规则管理
 *
 * @author Michael Li
 * 2020-10-11
 */
@RestController
@RequestMapping("/sys/platform/numbering")
@Api(value = "发号器-规则管理", tags = {"发号器-规则管理"})
public class SysPlatformNumberRuleController {

    private final ISysPlatformNumberRuleService iSysPlatformNumberRuleService;

    public SysPlatformNumberRuleController(ISysPlatformNumberRuleService sysNumberRuleService) {
        this.iSysPlatformNumberRuleService = sysNumberRuleService;
    }

    /**
     * 创建发号器规则
     *
     * @param rule 发号器规则
     * @return 创建结果
     */
    @PostMapping("/rules")
    @ApiOperation("创建发号器规则，成功则返回对应ID")
    public ApiResult<Long> create(@RequestBody @Valid SysPlatformNumberRuleAddParam rule) {
        return iSysPlatformNumberRuleService.create(rule);
    }

    /**
     * 更新发号器规则
     *
     * @param rule 发号器规则
     * @return 更新结果
     */
    @PutMapping("/rules")
    @ApiOperation("更新发号器规则")
    public ApiResult<Long> update(@RequestBody @Valid SysPlatformNumberRuleAddParam rule) {
        return iSysPlatformNumberRuleService.update(rule);
    }

    /**
     * 检索发号器规则
     *
     * @param param 查询参数
     * @return 发号器规则
     */
    @PostMapping("/q")
    @ApiOperation("检索发号器规则")
    public ApiResult<PagingVO<SysPlatformNumberRuleVO>> search(@RequestBody SysPlatformNumberRuleQParam param) {
        return iSysPlatformNumberRuleService.search(param);
    }

    /**
     * 删除发号器规则
     *
     * @param id 发号器规则ID
     * @return 删除结果
     */
    @DeleteMapping("/rules/{id}")
    @ApiOperation("删除发号器规则")
    public ApiResult<Boolean> remove(@PathVariable("id") Long id) {
        return iSysPlatformNumberRuleService.removeById(id);
    }
//
//    /**
//     * 生成样例号码
//     *
//     * @param param 发号参数
//     * @return 样例号码
//     */
//    @PostMapping("/samplecode")
//    @ApiOperation("生成样例号码")
//    public ApiResult<String> getSampleCode(@RequestBody @Valid SysNumberReqParam param) {
//        return iSysPlatformNumberRuleService.generateSampleCode(param.getRuleId(), param.getRuntimeValue());
//    }

//    /**
//     * 生成号码
//     *
//     * @param param 参数
//     * @return 号码
//     */
//    @PostMapping("/code")
//    @ApiOperation("生成号码")
//    public ApiResult<String> getCode(@RequestBody SysNumberReqParam param) {
//        return iSysPlatformNumberRuleService.generateCode(param.getRuleId(), param.getRuntimeValue());
//    }

    /**
     * 获取规则详细信息
     *
     * @param id 规则ID
     * @return 规则详细
     */
    @GetMapping("/rules/{id}")
    @ApiOperation("根据规则ID，获取规则对象，包括规则明细列表")
    public ApiResult<SysPlatformNumberRuleVO> one(@PathVariable("id") Long id) {
        return iSysPlatformNumberRuleService.oneCombined(id);
    }

    /**
     * 更新启用状态
     * <p>
     * 自动取反，如果当前是启用，则修改为禁用，反之修改为启用；
     *
     * @param id 发号规则ID
     * @return 发号规则ID
     */
    @ApiOperation(value = "更新启用状态", notes = "自动取反，如果当前是启用，则修改为禁用，反之修改为启用")
    @ApiImplicitParam(name = "id", value = "发号规则ID", required = true)
    @PatchMapping("/update/{id}/enabled")
    public ApiResult<Long> updateEnabled(@PathVariable Long id) {
        return iSysPlatformNumberRuleService.updateEnabled(id);
    }

    /**
     * 取号类型
     *
     * @return 取号类型列表
     */
    @ApiOperation(value = "取号类型")
    @ApiOperationSupport(order = 7)
    @GetMapping(value = "/numType")
    public ApiResult<List<CodeNameParam>> numType() {
        var values = Arrays.stream(SysNumType.values())
                .map(t -> new CodeNameParam(t.name(), t.getDescription()))
                .collect(Collectors.toList());
        return ApiResult.ok(values);
    }
}
