package com.elitescloud.cloudt.platform.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitescloud.cloudt.platform.convert.SysPlatformNumberRuleConvert;
import com.elitescloud.cloudt.platform.model.entity.SysPlatformNumberRuleDO;
import com.elitescloud.cloudt.platform.model.params.seq.SysPlatformNumberRuleAddParam;
import com.elitescloud.cloudt.platform.model.params.seq.SysPlatformNumberRuleDtlVO;
import com.elitescloud.cloudt.platform.model.params.seq.SysPlatformNumberRuleQParam;
import com.elitescloud.cloudt.platform.model.params.seq.SysPlatformNumberRuleVO;
import com.elitescloud.cloudt.platform.service.ISysPlatformNumberRuleDtlService;
import com.elitescloud.cloudt.platform.service.ISysPlatformNumberRuleService;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformNumberRuleDtlRepoProc;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformNumberRuleRepo;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformNumberRuleRepoProc;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/6/17
 */
@Service
@Log4j2
public class ISysPlatformNumberRuleServiceImpl implements ISysPlatformNumberRuleService {

    private static final SysPlatformNumberRuleConvert CONVERT = SysPlatformNumberRuleConvert.INSTANCE;
    private static final Map<String, DateTimeFormatter> FORMATTER_MAP = new HashMap<>();

    @Autowired
    private SysPlatformNumberRuleRepo sysPlatformNumberRuleRepo;
    @Autowired
    private SysPlatformNumberRuleRepoProc sysPlatformNumberRuleRepoProc;
    @Autowired
    private SysPlatformNumberRuleDtlRepoProc sysPlatformNumberRuleDtlRepoProc;
    @Autowired
    private ISysPlatformNumberRuleDtlService sysNumberRuleDtlService;
//    @Autowired
//    private ISysNextNumberService sysNextNumberService;

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> create(SysPlatformNumberRuleAddParam rule) {


        boolean exists = sysPlatformNumberRuleRepoProc.existsRuleCode(rule.getRuleCode());
        if (exists) {
            return ApiResult.fail("编号规则：" + rule.getRuleCode() + "， 已经存在");
        }
        SysPlatformNumberRuleDO ruleDO = CONVERT.voToDO(rule);
        if (ruleDO.getEnabled() == null) {
            ruleDO.setEnabled(true);
        }

        sysPlatformNumberRuleRepo.save(ruleDO);
        return ApiResult.ok(ruleDO.getId());
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> update(SysPlatformNumberRuleAddParam rule) {
        // 校验数据
        if (rule.getId() == null) {
            return ApiResult.fail("数据标识ID为空");
        }
        SysPlatformNumberRuleDO numberRuleDO = sysPlatformNumberRuleRepoProc.get(rule.getId());
        if (numberRuleDO == null) {
            return ApiResult.fail("修改的数据不存在");
        }
        boolean exists = sysPlatformNumberRuleRepoProc.existsRuleCode(rule.getRuleCode(), rule.getId());
        if (exists) {
            return ApiResult.fail("编号规则：" + rule.getRuleCode() + "， 已经存在");
        }

        // 更新数据
        CONVERT.copyAddVoToDO(rule, numberRuleDO);
        if (numberRuleDO.getEnabled() == null) {
            numberRuleDO.setEnabled(true);
        }
        sysPlatformNumberRuleRepo.save(numberRuleDO);

        return ApiResult.ok(numberRuleDO.getId());
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public ApiResult<Long> updateEnabled(Long id) {
        var enabled = sysPlatformNumberRuleRepoProc.getEnabled(id);
        enabled = enabled == null || !enabled;

        sysPlatformNumberRuleRepoProc.updateEnabled(id, enabled);
        return ApiResult.ok(id);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Boolean> removeById(Long id) {
        if (id == null) {
            return ApiResult.fail("ID为空");
        }

        sysPlatformNumberRuleRepoProc.delete(id);
        return ApiResult.ok(true);
    }

    @Override
    @SysCodeProc
    public ApiResult<PagingVO<SysPlatformNumberRuleVO>> search(SysPlatformNumberRuleQParam param) {
        PagingVO<SysPlatformNumberRuleVO> pageResult = sysPlatformNumberRuleRepoProc.queryForPage(param).map(CONVERT::doToVO);
        return ApiResult.ok(pageResult);
    }

    @Override
    public ApiResult<String> generateSampleCode(Long ruleId, List<String> runtimeValues) {
        // 生成编号
      //  String code = executeGenerateCode(ruleId, d -> 1L, runtimeValues);

        return ApiResult.ok();
    }

    @Override
    public ApiResult<String> generateSampleCode(String ruleCode, List<String> runtimeValues) {
        if (!StringUtils.hasText(ruleCode)) {
            return ApiResult.fail("编号为空");
        }
        Long id = sysPlatformNumberRuleRepoProc.getIdByCode(ruleCode);
        if (id == null) {
            throw new BusinessException("未查询到有效的发号规则[" + ruleCode + "]");
        }

        return generateSampleCode(id, runtimeValues);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<String> generateCode(Long ruleId, List<String> runtimeValues) {
      //  String code = executeGenerateCode(ruleId, d -> sysNextNumberService.getNextNumber(d.getNumberPattern(), d.getNnLen()).getData(), runtimeValues);

        return ApiResult.ok();
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<String> generateCode(String ruleCode, List<String> runtimeValues) {
        Long id = sysPlatformNumberRuleRepoProc.getIdByCode(ruleCode);
        if (id == null) {
            throw new BusinessException("未查询到有效的发号规则[" + ruleCode + "]");
        }
        return generateCode(id, runtimeValues);
    }

    @Override
    public ApiResult<SysPlatformNumberRuleVO> oneCombined(Long id) {
        if (id == null) {
            return ApiResult.fail("规则ID为空");
        }
        SysPlatformNumberRuleDO numberRule = sysPlatformNumberRuleRepoProc.get(id);
        if (numberRule == null) {
            return ApiResult.fail("数据不存在");
        }

        SysPlatformNumberRuleVO ruleVO = CONVERT.doToVO(numberRule);
        List<SysPlatformNumberRuleDtlVO> details = sysNumberRuleDtlService.listByRuleId(id).getData();
        ruleVO.setRuleDetails(ObjectUtil.defaultIfNull(details, Collections.emptyList()));

        return ApiResult.ok(ruleVO);
    }

//    private String executeGenerateCode(Long ruleId, Function<SysNumberRuleDtlDTO, Long> nextNumberFunction,
//                                       List<String> runtimeValues) {
//        if (ruleId == null) {
//            throw new BusinessException("发号器规则ID为空");
//        }
//
//        // 查询发号规则
//        List<SysNumberRuleDtlDTO> ruleDetails = sysNumberRuleDtlRepoProc.queryByRuleId(ruleId);
//        if (ruleDetails.isEmpty()) {
//            throw new BusinessException("未查询到发号规则配置");
//        }
//
//        // 生成编号
//        return executeGenerateCode(ruleId, ruleDetails, nextNumberFunction, runtimeValues);
//    }
//
//    private String executeGenerateCode(@NonNull Long ruleId, @NonNull List<SysNumberRuleDtlDTO> ruleDetails,
//                                       Function<SysNumberRuleDtlDTO, Long> nextNumberFunction,
//                                       List<String> runtimeValues) {
//        StringBuilder code = new StringBuilder();
//
//        int offset = 0;
//        for (SysNumberRuleDtlDTO ruleDetail : ruleDetails) {
//            // 取号类型
//            if (!StringUtils.hasText(ruleDetail.getNumberType())) {
//                throw new BusinessException("发号器规则【" + ruleId + "】配置有误，取号类型为空");
//            }
//
//            // 根据不同取号类型生成号码
//            LocalDateTime now = LocalDateTime.now();
//            String pattern = ruleDetail.getNumberPattern();
//            switch (ruleDetail.getNumberType()) {
//                case "FS": {
//                    // 固定值
//                    if (!StringUtils.hasText(pattern)) {
//                        throw new BusinessException("发号器规则【" + ruleId + "】配置有误，固定值的模式为空");
//                    }
//                    code.append(pattern);
//                    break;
//                }
//                case "DP": {
//                    // 日期格式
//                    if (!StringUtils.hasText(pattern)) {
//                        throw new BusinessException("发号器规则【" + ruleId + "】配置有误，日期格式的模式为空");
//                    }
//                    code.append(FORMATTER_MAP.computeIfAbsent(pattern, DateTimeFormatter::ofPattern).format(now));
//                    break;
//                }
//                case "NN": {
//                    // 自增
//                    Integer len = ruleDetail.getNnLen();
//                    if (len == null) {
//                        throw new BusinessException("发号器规则【" + ruleId + "】配置有误，自增序号宽度为空");
//                    }
//                    Long nextNumber = nextNumberFunction.apply(ruleDetail);
//                    code.append(preZero(nextNumber, len));
//                    break;
//                }
//                default:
//                    if (runtimeValues != null && offset < runtimeValues.size()) {
//                        if (CharSequenceUtil.equals("NN", pattern)) {
//                            Integer len = ruleDetail.getNnLen();
//                            if (len == null) {
//                                throw new BusinessException("发号器规则【" + ruleId + "】配置有误，自增序号宽度为空");
//                            }
//                            Long nextNumber = sysNextNumberService.getNextNumberForRuntime(runtimeValues.get(offset), len).getData();
//                            code.append(preZero(nextNumber, len));
//                        } else {
//                            code.append(runtimeValues.get(offset));
//                        }
//                        offset++;
//                    }
//            }
//        }
//
//        return code.toString();
//    }

    private String preZero(long val, int len) {
        String format = "%0" + len + "d";
        return String.format(format, val);
    }
}
