package com.elitescloud.cloudt.platform.service.impl;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.extra.pinyin.PinyinUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.constant.AreaType;
import com.elitescloud.cloudt.context.util.TreeDataUtil;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitescloud.cloudt.platform.convert.SysPlatformAreaConvert;
import com.elitescloud.cloudt.platform.model.entity.SysPlatformAreaDO;
import com.elitescloud.cloudt.platform.model.vo.extend.resp.AreaDetailRespVO;
import com.elitescloud.cloudt.platform.model.vo.extend.resp.AreaMngTreeRespVO;
import com.elitescloud.cloudt.platform.model.vo.extend.save.PlatformAreaSaveVO;
import com.elitescloud.cloudt.platform.service.SysPlatformAreaService;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformAreaRepo;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformAreaRepoProc;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.Comparator;
import java.util.List;
import java.util.Map;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2023/1/17
 */
@Service
@Log4j2
public class SysPlatformAreaServiceImpl extends BaseServiceImpl implements SysPlatformAreaService {
    private static final SysPlatformAreaConvert CONVERT = SysPlatformAreaConvert.INSTANCE;
    @Autowired
    private SysPlatformAreaRepo areaRepo;
    @Autowired
    private SysPlatformAreaRepoProc areaRepoProc;

    @Override
    public ApiResult<List<AreaMngTreeRespVO>> tree(Boolean tree) {
        var areaList = areaRepoProc.getTree(null, null, null, t -> {
            var respVO = new AreaMngTreeRespVO();
            respVO.setEnabled(t.getEnabled());
            respVO.setPinyin(t.getPinyin());
            respVO.setId(t.getId());
            respVO.setCode(t.getAreaCode());
            respVO.setName(t.getAreaName());
            respVO.setSortNo(t.getSortNo());
            respVO.setParentId(t.getPId());
            respVO.setParentCode(t.getParentAreaCode());

            return respVO;
        });

        // 是否转树
        if (!ObjectUtil.defaultIfNull(tree, true) || areaList.isEmpty()) {
            return ApiResult.ok(areaList);
        }
        TreeDataUtil<AreaMngTreeRespVO> treeDataUtil = new TreeDataUtil<AreaMngTreeRespVO>(areaList, AreaMngTreeRespVO::getId,
                AreaMngTreeRespVO::getParentId, AreaMngTreeRespVO::setChildren, Comparator.comparingInt(AreaMngTreeRespVO::getSortNo));
        var result = (List<AreaMngTreeRespVO>) treeDataUtil.getRoots();
        return ApiResult.ok(result);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(PlatformAreaSaveVO saveVO) {
        SysPlatformAreaDO areaDO = null;
        try {
            areaDO = saveVO.getId() == null ? checkForInsert(saveVO) : checkForUpdate(saveVO);
        } catch (IllegalArgumentException e) {
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        // 默认值处理
        Long parentId = StringUtils.hasText(areaDO.getParentAreaCode()) ? areaRepoProc.getId(areaDO.getParentAreaCode()) : null;
        if (areaDO.getEnabled() == null) {
            areaDO.setEnabled(true);
        }
        if (areaDO.getSortNo() == null) {
            areaDO.setSortNo(areaRepoProc.findMaxSortNo(parentId) + 1);
        }

        String oldCodePath = saveVO.getId() == null ? null : areaRepoProc.getCodePath(saveVO.getId());
        areaRepo.save(areaDO);

        // 保存树节点信息
        areaRepoProc.saveTreeNode(areaDO, parentId, areaDO.getSortNo());
        // 如果codePath有调整，则修改下级
        if (oldCodePath != null && !StrUtil.equals(oldCodePath, areaDO.getCodePath())) {
            var idAndCodePathMap = areaRepoProc.getIdAndCodePathByCodePath(oldCodePath);
            var len = oldCodePath.length();
            for (Map.Entry<Long, String> entry : idAndCodePathMap.entrySet()) {
                String newPath = areaDO.getCodePath() + entry.getValue().substring(len);
                areaRepoProc.updateCodePath(entry.getKey(), newPath);
            }
        }

        return ApiResult.ok(areaDO.getId());
    }

    @Override
    public ApiResult<AreaDetailRespVO> get(Long id) {
        Assert.notNull(id, "ID为空");

        return areaRepoProc.getOptional(id)
                .map(t -> {
                    var respVO = CONVERT.do2DetailRespVO(t);
                    // 父节点
                    if (StringUtils.hasText(t.getParentAreaCode())) {
                        respVO.setParentAreaName(areaRepoProc.getAreaName(t.getParentAreaCode()));
                    }
                    // 节点类型
                    if (StringUtils.hasText(t.getAreaType())) {
                        respVO.setAreaTypeName(super.udcValue(new AreaType(t.getAreaType())));
                    }
                    return respVO;
                }).map(ApiResult::ok)
                .orElse(ApiResult.fail("数据不存在"));
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        Assert.notNull(id, "ID为空");

        var data = areaRepoProc.get(id);
        if (data == null) {
            return ApiResult.ok(id);
        }

        areaRepoProc.removeTreeNode(data);
        areaRepoProc.delete(id);
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> execReBuildTree() {
        // 获取编码与id关系
        var idCodeMap = areaRepoProc.getIdAndCode();
        if (idCodeMap.isEmpty()) {
            return ApiResult.ok();
        }

        // 开始重构
        areaRepoProc.rebuildTree(null,
                t -> StringUtils.hasText(t.getParentAreaCode()) ? idCodeMap.get(t.getParentAreaCode()) : null);
        return ApiResult.ok();
    }

    private SysPlatformAreaDO checkForInsert(PlatformAreaSaveVO saveVO) {
        var areaDO = CONVERT.saveVo2Do(saveVO);

        // 判断编码是否已存在
        var exists = areaRepoProc.existsAreaCode(areaDO.getAreaCode(), null);
        Assert.isTrue(!exists, "地区编码已存在");

        // 上级地区
        if (StringUtils.hasText(areaDO.getParentAreaCode())) {
            exists = areaRepoProc.existsAreaCode(areaDO.getParentAreaCode(), null);
            Assert.isTrue(exists, "上级地区编码不存在");

            String parentCodePath = areaRepoProc.getCodePath(areaDO.getParentAreaCode());
            areaDO.setCodePath(parentCodePath + SysPlatformAreaDO.PATH_SEPARATOR + areaDO.getAreaCode());
        } else {
            areaDO.setCodePath(SysPlatformAreaDO.PATH_SEPARATOR + areaDO.getAreaCode());
        }

        // 拼音
        areaDO.setPinyin(PinyinUtil.getPinyin(saveVO.getAreaName()));

        return areaDO;
    }

    private SysPlatformAreaDO checkForUpdate(PlatformAreaSaveVO saveVO) {
        var areaDO = areaRepoProc.get(saveVO.getId());
        Assert.notNull(areaDO, "修改的数据不存在");

        // 判断编码是否已存在
        if (!StrUtil.equals(saveVO.getAreaCode(), areaDO.getAreaCode())) {
            var exists = areaRepoProc.existsAreaCode(saveVO.getAreaCode(), null);
            Assert.isTrue(!exists, "地区编码已存在");
        }

        if (StrUtil.isBlank(saveVO.getParentAreaCode())) {
            // 上级为空
            areaDO.setCodePath(SysPlatformAreaDO.PATH_SEPARATOR + areaDO.getAreaCode());
        } else if (!StrUtil.equals(saveVO.getParentAreaCode(), areaDO.getParentAreaCode())) {
            // 上级有调整
            var exists = areaRepoProc.existsAreaCode(saveVO.getParentAreaCode(), null);
            Assert.isTrue(exists, "上级地区编码不存在");

            String parentCodePath = areaRepoProc.getCodePath(areaDO.getParentAreaCode());
            areaDO.setCodePath(parentCodePath + SysPlatformAreaDO.PATH_SEPARATOR + areaDO.getAreaCode());
        }

        // 拼音
        if (!StrUtil.equals(saveVO.getAreaName(), areaDO.getAreaName()) || StrUtil.isBlank(areaDO.getAreaName())) {
            areaDO.setPinyin(PinyinUtil.getPinyin(saveVO.getAreaName()));
        }

        CONVERT.saveVo2Do(saveVO, areaDO);
        return areaDO;
    }
}
