package com.elitescloud.cloudt.platform.service.repo;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.platform.model.entity.QSysMenuOperationDO;
import com.elitescloud.cloudt.platform.model.entity.QSysPlatformMenusDO;
import com.elitescloud.cloudt.platform.model.entity.SysMenuOperationDO;
import com.elitescloud.cloudt.platform.model.vo.resp.MenuBasicRespVO;
import com.elitescloud.cloudt.platform.model.vo.resp.MenuOperationRespVO;
import com.elitescloud.cloudt.system.service.model.entity.QSysBusinessObjectDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysBusinessOperationDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysPlatformAppDO;
import com.querydsl.core.types.Expression;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.JPAExpressions;
import org.springframework.stereotype.Repository;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/3/21
 */
@Repository
public class MenuOperationRepoProc extends BaseRepoProc<SysMenuOperationDO> {
    private static final QSysMenuOperationDO QDO = QSysMenuOperationDO.sysMenuOperationDO;
    private static final QSysPlatformMenusDO QDO_MENU = QSysPlatformMenusDO.sysPlatformMenusDO;
    private static final QSysPlatformAppDO QDO_APP = QSysPlatformAppDO.sysPlatformAppDO;
    private static final QSysBusinessObjectDO QDO_OBJECT = QSysBusinessObjectDO.sysBusinessObjectDO;
    private static final QSysBusinessOperationDO QDO_OPERATION = QSysBusinessOperationDO.sysBusinessOperationDO;

    public MenuOperationRepoProc() {
        super(QDO);
    }

    public void deleteByMenu(@NotBlank String menuCode) {
        super.delete(QDO.menusCode.eq(menuCode));
    }

    public void delete(@NotBlank String menuCode, @NotEmpty Collection<String> operationCodes) {
        super.delete(QDO.operationCode.in(operationCodes).and(QDO.menusCode.eq(menuCode)));
    }

    public List<String> getOperationCodesByMenuCode(@NotBlank String menuCode) {
        return super.getValueListByValue(QDO.operationCode, QDO.menusCode, menuCode);
    }

    public Map<String, Long> countMenuOfOperationByBusinessObjectId(@NotEmpty Collection<Long> businessObjectIds) {
        var countExpression = QDO.count();
        return super.jpaQueryFactory.select(QDO.operationCode, countExpression)
                .from(QDO)
                .where(QDO.operationCode.in(
                        JPAExpressions.select(QDO_OPERATION.operationCode)
                                .from(QDO_OPERATION)
                                .where(QDO_OPERATION.businessObjectId.in(businessObjectIds))
                )).groupBy(QDO.operationCode)
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.operationCode), t -> t.get(countExpression), (t1, t2) -> t1));
    }

    public List<MenuBasicRespVO> listMenusOfOperation(@NotBlank String operationCode) {
        var qBean = Projections.bean(MenuBasicRespVO.class, QDO_MENU.id, QDO_MENU.menusAppCode, QDO_APP.appName.as("menusAppName"),
                QDO_MENU.menusName, QDO_MENU.menusType, QDO_MENU.nodeType, QDO_MENU.menusCode, QDO_MENU.menusRoute, QDO_MENU.menusDescribe);
        return super.jpaQueryFactory.select(qBean)
                .from(QDO_MENU)
                .leftJoin(QDO).on(QDO.menusCode.eq(QDO_MENU.menusCode))
                .leftJoin(QDO_APP).on(QDO_APP.appCode.eq(QDO.appCode))
                .where(QDO.operationCode.eq(operationCode))
                .fetch();
    }

    public List<MenuOperationRespVO> listOperationByMenuCode(@NotBlank String menuCode) {
        return this.listMenuOperation(null, menuCode);
    }

    /**
     * 查询菜单下的操作
     *
     * @param appCodes
     * @return
     */
    public Map<String, List<MenuOperationRespVO>> listOperationOfMenu(Collection<String> appCodes) {
        return this.listMenuOperation(appCodes, null)
                .stream()
                .collect(Collectors.groupingBy(MenuOperationRespVO::getMenusCode));
    }

    private List<MenuOperationRespVO> listMenuOperation(Collection<String> appCodes, String menuCode) {
        var subQueryMenuName = JPAExpressions.select(QDO_MENU.menusName).from(QDO_MENU).where(QDO_MENU.menusCode.eq(QDO.menusCode));
        var qBean = new Expression[]{QDO.id, QDO.appCode, QDO_APP.appName, QDO.menusCode, subQueryMenuName,
                QDO_OPERATION.businessObjectCode, QDO_OBJECT.name, QDO_OBJECT.customName,
                QDO.operationCode, QDO_OPERATION.customName, QDO_OPERATION.operationDescription, QDO_OPERATION.apiMethod,
                QDO_OPERATION.apiUrl};
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.menusCode, menuCode)
                .andIn(QDO_OPERATION.appCode, appCodes)
                .and(QDO_OPERATION.id.isNotNull())
                .build();

        return super.jpaQueryFactory.select(qBean)
                .from(QDO)
                .leftJoin(QDO_APP).on(QDO_APP.appCode.eq(QDO.appCode))
                .leftJoin(QDO_OPERATION).on(QDO_OPERATION.operationCode.eq(QDO.operationCode))
                .leftJoin(QDO_OBJECT).on(QDO_OBJECT.code.eq(QDO_OPERATION.businessObjectCode))
                .where(predicate)
                .orderBy(QDO.createTime.desc())
                .fetch()
                .stream().map(t -> {
                    MenuOperationRespVO respVO = new MenuOperationRespVO();
                    respVO.setId(t.get(QDO.id));
                    respVO.setAppCode(t.get(QDO.appCode));
                    respVO.setAppName(t.get(QDO_APP.appName));
                    respVO.setMenusCode(t.get(QDO.menusCode));
                    respVO.setMenuName(t.get(subQueryMenuName));
                    respVO.setBusinessObjectCode(t.get(QDO_OPERATION.businessObjectCode));
                    respVO.setBusinessObjectName(CharSequenceUtil.blankToDefault(t.get(QDO_OBJECT.customName), t.get(QDO_OBJECT.name)));
                    respVO.setOperationCode(t.get(QDO.operationCode));
                    respVO.setOperationName(CharSequenceUtil.blankToDefault(t.get(QDO_OPERATION.customName), t.get(QDO_OPERATION.operationDescription)));
                    respVO.setApiMethod(t.get(QDO_OPERATION.apiMethod));
                    respVO.setApiUrl(t.get(QDO_OPERATION.apiUrl));

                    return respVO;
                }).collect(Collectors.toList());
    }
}
