package com.elitescloud.cloudt.platform.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.platform.convert.UdcConvert;
import com.elitescloud.cloudt.platform.model.entity.QSysPlatformUdcDO;
import com.elitescloud.cloudt.platform.model.params.udc.*;
import com.elitescloud.cloudt.platform.model.vo.SysPlatformUdcVO;
import com.elitescloud.cloudt.platform.model.vo.SysPlatformUdcValueVO;
import com.elitescloud.cloudt.platform.service.SysPlatformUdcService;
import com.elitescloud.cloudt.platform.service.repo.*;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.dsl.Expressions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * @Description:
 * @author: niu.chen
 * @date: 2022.09.15
 **/
@Service
@Slf4j
@RequiredArgsConstructor
public class SysPlatformUdcServiceImpl extends BaseServiceImpl implements SysPlatformUdcService {
    private final SysPlatformUdcRepo sysPlatformUdcRepo;
    private final SysPlatformUdcValueRepoProc sysPlatformUdcValueRepoProc;
    private final SysPlatformUdcValueRepo sysPlatformUdcValueRepo;
    private final SysPlatformAppRepo sysPlatformAppRepo;
    private final UdcProvider udcProvider;
    private final TaskExecutor taskExecutor;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> addUdc(AddUdcParam addUdcParam) {
        var appList = sysPlatformAppRepo.findByAppCode(addUdcParam.getAppCode());
        if (appList.size() != 1) {
            return ApiResult.fail(appList.isEmpty() ? "应用编码不存在" : "应用编码异常，存在多个");
        }

        var count = sysPlatformUdcRepo.countAllByAppCodeAndUdcCode(addUdcParam.getAppCode(), addUdcParam.getUdcCode());
        if (count != 0) {
            return ApiResult.fail("应用UDC编码重复");
        }
        var udcDo = UdcConvert.INSTANCE.saveParamToDo(addUdcParam);

//        if (addUdcParam.getSysPlatformUdcValueVOList() != null && addUdcParam.getSysPlatformUdcValueVOList().size() > 0) {
//            var udcValueList = addUdcParam.getSysPlatformUdcValueVOList().stream().map(
//                    UdcConvert.INSTANCE::saveParamToDo
//            ).collect(Collectors.toList());
//            sysPlatformUdcValueRepo.saveAll(udcValueList);
//        }
        return ApiResult.ok(sysPlatformUdcRepo.save(udcDo).getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> addUdcValue(AddUdcValueParam addUdcValueParam) {
        var udcDo = sysPlatformUdcRepo.findAllByAppCodeAndUdcCode(addUdcValueParam.getAppCode(),
                addUdcValueParam.getUdcCode());
        if (udcDo == null) {
            return ApiResult.fail("应用APP的UDC编码不存在," + addUdcValueParam.getAppCode() + "," + addUdcValueParam.getUdcCode());
        }
        var count = sysPlatformUdcValueRepo.countAllByAppCodeAndUdcCodeAndUdcValueCode(addUdcValueParam.getAppCode(),
                addUdcValueParam.getUdcCode(), addUdcValueParam.getUdcValueCode());
        if (count != 0) {
            return ApiResult.fail("UDC编码重复");
        }
        var udcValueDo = UdcConvert.INSTANCE.saveParamToDo(addUdcValueParam);
        udcValueDo.setAllowDefault(true);
        udcValueDo= sysPlatformUdcValueRepo.save(udcValueDo);
        // 清掉缓存
        this.clearCache(addUdcValueParam.getAppCode(), udcValueDo.getUdcCode());
        return ApiResult.ok(udcValueDo.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateUdcValue(UdcValueSaveVO saveVO) {
        Assert.notNull(saveVO.getId(), "ID为空");
        var udcValueDO = sysPlatformUdcValueRepoProc.get(saveVO.getId());
        Assert.notNull(udcValueDO, "修改的数据不存在");

        saveVO.setUdcOrder(ObjectUtil.defaultIfNull(saveVO.getUdcOrder(), 0));
        saveVO.setAllowStart(ObjectUtil.defaultIfNull(saveVO.getAllowStart(), true));
        UdcConvert.INSTANCE.copySaveVo2Do(saveVO, udcValueDO);

        sysPlatformUdcValueRepo.save(udcValueDO);
        return ApiResult.ok(saveVO.getId());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Boolean> deleteFlagUdc(Long id) {
        sysPlatformUdcRepo.findById(id).ifPresentOrElse(
                sysPlatformUdcDO -> {
                    sysPlatformUdcDO.setDeleteFlag(1);
                    sysPlatformUdcRepo.save(sysPlatformUdcDO);

                    sysPlatformUdcValueRepo.findAllByAppCodeAndUdcCode(sysPlatformUdcDO.getAppCode(),
                                    sysPlatformUdcDO.getUdcCode())
                            .forEach(sysPlatformUdcValueDO -> {
                                sysPlatformUdcValueDO.setDeleteFlag(1);
                                sysPlatformUdcValueRepo.save(sysPlatformUdcValueDO);
                            });

                    // 清掉缓存
                    this.clearCache(sysPlatformUdcDO.getAppCode(), null);
                },
                () -> {
                    throw new BusinessException("id不存在");
                }
        );

        return ApiResult.ok(true);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Boolean> deleteFlagUdcValue(Long id) {
        sysPlatformUdcValueRepo.findById(id).ifPresentOrElse(sysPlatformUdcValueDO -> {
                    sysPlatformUdcValueDO.setDeleteFlag(1);
                    sysPlatformUdcValueRepo.save(sysPlatformUdcValueDO);

                    // 清掉缓存
                    this.clearCache(sysPlatformUdcValueDO.getAppCode(), sysPlatformUdcValueDO.getUdcCode());
                }
                , () -> {
                    throw new BusinessException("id不存在");
                });
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> deleteUdc(Long id) {
        var udcDoOp = sysPlatformUdcRepo.findById(id);
        if (udcDoOp.isPresent()) {
            var udcDo = udcDoOp.get();
            var list = sysPlatformUdcValueRepo.findAllByAppCodeAndUdcCode(udcDo.getAppCode(), udcDo.getUdcCode());
            if (list.size() > 0) {
                return ApiResult.fail("请先删除全部UDC子项");
            } else {
                sysPlatformUdcRepo.deleteById(id);
                this.clearCache(udcDo.getAppCode(), udcDo.getUdcCode());

                // 删除租户下的
//                this.applyForAllTenant(() -> {
//                    // 清掉缓存
//                    this.clearCache(udcDo.getAppCode(), udcDo.getUdcCode());
//
//                    sysPlatformUdcRepoProc.deleteByAppCodeAndUdcCode(udcDo.getAppCode(), udcDo.getUdcCode());
//                    return null;
//                });
            }
        }

        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> deleteUdcValue(Long id) {
        var valueDO = sysPlatformUdcValueRepoProc.get(id);
        if (valueDO == null) {
            return ApiResult.noData();
        }
        sysPlatformUdcValueRepo.deleteById(id);

        // 清掉缓存
        this.clearCache(valueDO.getAppCode(), valueDO.getUdcCode());

        // 删除租户下的
//        this.applyForAllTenant(() -> {
//            // 清掉缓存
//            this.clearCache(valueDO.getAppCode(), valueDO.getUdcCode());
//
//            sysPlatformUdcValueRepoProc.delete(valueDO.getAppCode(), valueDO.getUdcCode(), valueDO.getUdcValueCode(), false);
//            return null;
//        });
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> updateUdc(Long id, UpdateUdcParam updateUdcParam) {
        sysPlatformUdcRepo.findById(id).ifPresentOrElse(
                sysPlatformUdcDO -> {
//                    sysPlatformUdcDO.setAppCode(updateUdcParam.getAppCode());
//                    sysPlatformUdcDO.setUdcCode(updateUdcParam.getUdcCode());
                    sysPlatformUdcDO.setUdcName(updateUdcParam.getUdcName());
                    sysPlatformUdcDO.setAllowUpdate(updateUdcParam.getAllowUpdate());
                    sysPlatformUdcDO.setAllowAddValue(updateUdcParam.getAllowAddValue());
                    sysPlatformUdcDO.setUdcDescribe(updateUdcParam.getUdcDescribe());
                    sysPlatformUdcDO.setParentUdcCode(updateUdcParam.getParentUdcCode());
                    sysPlatformUdcRepo.save(sysPlatformUdcDO);

                    // 清掉缓存
                    this.clearCache(sysPlatformUdcDO.getAppCode(), sysPlatformUdcDO.getUdcCode());
                }
                , () -> {
                    throw new BusinessException("id不存在");
                });
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> updateUdcValueAllowStart(Long id, Boolean allowStart) {
        sysPlatformUdcValueRepo.findById(id).ifPresentOrElse(
                sysPlatformUdcDO -> {
                    sysPlatformUdcDO.setAllowStart(allowStart);
                    sysPlatformUdcValueRepo.save(sysPlatformUdcDO);

                    // 清掉缓存
                    this.clearCache(sysPlatformUdcDO.getAppCode(), sysPlatformUdcDO.getUdcCode());
                }
                , () -> {
                    throw new BusinessException("id不存在");
                });
        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<SysPlatformUdcVO> getUdc(Long id) {
        AtomicReference<SysPlatformUdcVO> atomicUdcVO = new AtomicReference<SysPlatformUdcVO>();
        sysPlatformUdcRepo.findById(id).ifPresentOrElse(
                sysPlatformUdcDO -> {
                    SysPlatformUdcVO sysPlatformUdcVO1 = UdcConvert.INSTANCE.selectDOToVO(sysPlatformUdcDO);
                    var udcValueVOList = sysPlatformUdcValueRepo
                            .findAllByAppCodeAndUdcCode(sysPlatformUdcDO.getAppCode(),
                                    sysPlatformUdcDO.getUdcCode()).stream()
                            .map(UdcConvert.INSTANCE::selectDOToVO)
                            .sorted(comparatorOfUdcValue())
                            .collect(Collectors.toList());
                    sysPlatformUdcVO1.setSysPlatformUdcValueVOList(udcValueVOList);
                    atomicUdcVO.set(sysPlatformUdcVO1);
                }
                , () -> {
                    throw new BusinessException("id不存在");
                });
        return ApiResult.ok(atomicUdcVO.get());
    }

    @Override
    public ApiResult<PagingVO<SysPlatformUdcVO>> queryUdc(QueryUdcParam queryUdcParam) {
        var qdo = QSysPlatformUdcDO.sysPlatformUdcDO;
        Predicate predicate = Expressions.booleanTemplate("1=1");
        predicate = StringUtils.isBlank(queryUdcParam.getAppCode()) ? predicate :
                ExpressionUtils.and(predicate, qdo.appCode.eq(queryUdcParam.getAppCode()));
        predicate = StringUtils.isBlank(queryUdcParam.getUdcName()) ? predicate :
                ExpressionUtils.and(predicate, qdo.udcName.like("%" + queryUdcParam.getUdcName() + "%"));
        predicate = StringUtils.isBlank(queryUdcParam.getUdcCode()) ? predicate :
                ExpressionUtils.and(predicate, qdo.udcCode.eq(queryUdcParam.getUdcCode()));

        var page = sysPlatformUdcRepo.findAll(predicate, queryUdcParam.getPageRequest());

        var pagingVo = PagingVO.<SysPlatformUdcVO>builder()
                .total(page.getTotalElements())
                .setRecords(page.get().map(UdcConvert.INSTANCE::selectDOToVO)
                        .collect(Collectors.toList()));
        return ApiResult.ok(pagingVo);

    }

    @Override
    public ApiResult<List<CodeNameParam>> getValueList(String appCode, String udcCode) {
        Assert.hasText(appCode, "应用编码为空");
        Assert.hasText(udcCode, "UDC编码为空");

        var result = sysPlatformUdcValueRepoProc.getValueList(appCode, udcCode, true);
        return ApiResult.ok(result);
    }

    private void clearCache(String appCode, String udcCode) {
        udcProvider.clearCache(appCode, udcCode);
    }

    private <T>void applyForAllTenant(Supplier<T> supplier) {
        CompletableFuture.runAsync(() -> tenantDataIsolateProvider.byAllTenant(supplier::get), taskExecutor)
                .whenComplete((res, e) -> {
                    if (e != null) {
                        log.error("执行异常：", e);
                    }
                });
    }

    private Comparator<SysPlatformUdcValueVO> comparatorOfUdcValue() {
        return Comparator.comparing(SysPlatformUdcValueVO::getUdcOrder, Comparator.nullsFirst(Integer::compareTo)).reversed();
    }
}
