package com.elitescloud.cloudt.platform.service.impl;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.DatetimeUtil;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.platform.convert.SysPlatformSqlExecuteConvert;
import com.elitescloud.cloudt.platform.model.constant.SqlTyeEnum;
import com.elitescloud.cloudt.platform.model.entity.QSysPlatformSqlExecuteDO;
import com.elitescloud.cloudt.platform.model.entity.SysPlatformSqlExecuteDO;
import com.elitescloud.cloudt.platform.model.params.sql.SysPlatformSqlExecuteParam;
import com.elitescloud.cloudt.platform.model.vo.SqlExecuteDataVo;
import com.elitescloud.cloudt.platform.service.SysPlatformSqlExecuteService;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformSqlExecuteRepo;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformSqlExecuteRepoProc;
import com.elitescloud.cloudt.system.service.SysAlertService;
import com.elitescloud.cloudt.system.service.util.JpaPredicateBuilder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.task.TaskExecutor;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.datasource.DriverManagerDataSource;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * @author : chen.niu
 * @description :
 * @date : 2023/10/25 12:57
 */
@Service
@Slf4j
public class SysPlatformSqlExecuteServiceImpl implements SysPlatformSqlExecuteService, InitializingBean {
    @Value("${sqlexecute.datasource.url:#{null}}")
    private String dbUrl;

    @Value("${sqlexecute.datasource.driver-class-name:#{null}}")
    private String dbDriverClassName;

    @Value("${sqlexecute.datasource.username:#{null}}")
    private String dbUsername;

    @Value("${sqlexecute.datasource.password:#{null}}")
    private String dbPassword;
    private JdbcTemplate jdbcTemplate;

    @Autowired
    private SysPlatformSqlExecuteRepo sqlExecuteRepo;
    @Autowired
    private SysPlatformSqlExecuteRepoProc repoProc;
    @Autowired
    private SysAlertService alertService;
    @Autowired
    private TaskExecutor taskExecutor;

    @Override
    public void afterPropertiesSet() throws Exception {
        jdbcTemplate = sqlExecuteDataJdbcTemplate();
    }

    public JdbcTemplate sqlExecuteDataJdbcTemplate() {
        if (dbUrl == null) {
            log.info("sql执行数据源配置空，不执行");
            return null;
        }
        DriverManagerDataSource dataSource = new DriverManagerDataSource();
        dataSource.setDriverClassName(dbDriverClassName);
        dataSource.setUrl(dbUrl);
        dataSource.setUsername(dbUsername);
        dataSource.setPassword(dbPassword);
        log.info("sql执行数据源=" + dbDriverClassName);
        log.info("sql执行数据源=" + dbUrl);
        log.info("sql执行数据源=" + dbUsername);
        log.info("sqlExecuteDataJdbcTemplate ok");
        return new JdbcTemplate(dataSource);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<String> importData(List<SqlExecuteDataVo> dataVoList) {
        try {
            for (SqlExecuteDataVo data : dataVoList) {
                sqlExecuteRepo.findBySqlCode(data.getSqlCode()).ifPresentOrElse(
                        sdo -> {
                            sdo.setSqlName(data.getSqlName());
                            sdo.setQuerySql(data.getQuerySql());
                            sdo.setCategory(data.getCategory());
                            sqlExecuteRepo.save(sdo);
                        }, () -> {
                            SysPlatformSqlExecuteDO sysPlatformSqlExecuteDO = new SysPlatformSqlExecuteDO();
                            sysPlatformSqlExecuteDO.setSqlName(data.getSqlName());
                            sysPlatformSqlExecuteDO.setQuerySql(data.getQuerySql());
                            sysPlatformSqlExecuteDO.setCategory(data.getCategory());
                            sysPlatformSqlExecuteDO.setSqlCode(data.getSqlCode());
                            sqlExecuteRepo.save(sysPlatformSqlExecuteDO);
                        }
                );
            }
            return ApiResult.ok("导入成功:" + dataVoList.size());
        } catch (Exception e) {
            log.error(e.getMessage());
            return ApiResult.fail("导入失败:" + e.getMessage());
        }
    }

    @Override
    public String getSqlByReplaceAll(String sql) {
        String result = sql.replaceAll("[\n\r]", " ");
        result = result.replaceAll("\t", " ");
        result = result.replaceAll("\n", " ");
        result = result.replaceAll("\r", " ");
        result = result.replaceAll("\\r", " ");
        result = result.replaceAll("[\\n\\r]", " ");
        result = result.replaceAll("\\t", " ");
        result = result.replaceAll("\\n", " ");
        result = result.replaceAll(" +", " ");
        result = result.replaceAll(" ", " ");
        result = result.replaceAll("[\u0000-\u001f]", " ");
        result = result.replaceAll("[\\u0000-\\u001f]", " ");
        return result;
    }

    @Override
    public ApiResult<PagingVO<SysPlatformSqlExecuteDO>> page(SysPlatformSqlExecuteParam queryVO) {
        var QDO = QSysPlatformSqlExecuteDO.sysPlatformSqlExecuteDO;
        var predicate = JpaPredicateBuilder.builder()
                .and(QDO.datasourceName::like, StringUtils.hasText(queryVO.getDatasourceName()) ? "%" + queryVO.getDatasourceName() + "%" : null)
                .and(QDO.sqlName::like, StringUtils.hasText(queryVO.getSqlName()) ? "%" + queryVO.getSqlName() + "%" : null)
                .and(QDO.querySql::like, StringUtils.hasText(queryVO.getQuerySql()) ? "%" + queryVO.getQuerySql() + "%" : null)
                .and(QDO.sqlCode::like, StringUtils.hasText(queryVO.getSqlCode()) ? "%" + queryVO.getSqlCode() + "%" : null)
                .and(QDO.category::eq, StringUtils.hasText(queryVO.getCategory()) ? queryVO.getCategory() : null)
                .getPredicate();
        var page = sqlExecuteRepo.findAll(predicate, queryVO.getPageRequest());
        var pagingVo = PagingVO.<SysPlatformSqlExecuteDO>builder()
                .total(page.getTotalElements())
                .setRecords(page.get().collect(Collectors.toList()));
        return ApiResult.ok(pagingVo);
    }

    @Override
    public ApiResult<Long> add(SysPlatformSqlExecuteParam addParam) {
        if (CharSequenceUtil.isNotBlank(addParam.getSqlType())) {
            var type = SqlTyeEnum.parse(addParam.getSqlType());
            if (type == null) {
                return ApiResult.fail("SQL类型不存在");
            }
        }
        Assert.notBlank(addParam.getSqlCode(), "SQL编码不能为空");
        var id = repoProc.getIdByCode(addParam.getSqlCode());
        if (id != null) {
            return ApiResult.fail("SQL编码已存在");
        }

        var addDo = SysPlatformSqlExecuteConvert.INSTANCE.voToDo(addParam);

        sqlExecuteRepo.save(addDo);
        return ApiResult.ok(addDo.getId());
    }

    @Override
    public ApiResult<Boolean> update(Long id, SysPlatformSqlExecuteParam updateParam) {
        Assert.notNull(id, "ID为空");
        Assert.notBlank(updateParam.getSqlCode(), "SQL编码不能为空");
        sqlExecuteRepo.findById(id).ifPresentOrElse(manageDO -> {
                   if (!CharSequenceUtil.equals(updateParam.getSqlCode(), manageDO.getSqlCode())) {
                       var existsId = repoProc.getIdByCode(updateParam.getSqlCode());
                       Assert.isNull(existsId, "SQL编码已存在");
                   }

                    manageDO.setQuerySql(updateParam.getQuerySql());
                    manageDO.setSqlName(updateParam.getSqlName());
                    manageDO.setDatasourceName(updateParam.getDatasourceName());

                    manageDO.setSqlCode(updateParam.getSqlCode());
                    manageDO.setCategory(updateParam.getCategory());
                    manageDO.setSqlType(updateParam.getSqlType());
                    manageDO.setLimitSize(updateParam.getLimitSize());
                    sqlExecuteRepo.save(manageDO);
                },
                () -> {
                    throw new BusinessException("id不存在");
                });
        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<SysPlatformSqlExecuteDO> findById(Long id) {
        var sqlDo = sqlExecuteRepo.findById(id);
        if (sqlDo.isPresent()) {
            return ApiResult.ok(sqlDo.get());
        } else {
            return ApiResult.ok();
        }
    }

    @Override
    public ApiResult<Boolean> deleteByIds(List<Long> ids) {
        sqlExecuteRepo.deleteAllById(ids);
        return ApiResult.ok();
    }

    @Override
    public ApiResult<SysPlatformSqlExecuteDO> executeQuery(Long id) {

        var sysPlatformSqlExecuteDOOptional = sqlExecuteRepo.findById(id);
        if (sysPlatformSqlExecuteDOOptional.isPresent()) {

            var sDo = sysPlatformSqlExecuteDOOptional.get();
            sDo.setExecuteStartTime(LocalDateTime.now());

            String sql = sDo.getQuerySql();
            Exception exp = null;
            int rowNums = 0;
            try {
                String result = getSqlByReplaceAll(sql);

                sDo.setQuerySql(result);
                var listMap = jdbcTemplate.queryForList(result);
                rowNums = listMap.size();
                sDo.setExecuteSqlResult(listMap.isEmpty() ? "0" : rowNums + "条");
            } catch (Exception e) {
                exp = e;
                log.error(e.getMessage());
                sDo.setExecuteSqlResult(e.getMessage());
            }

            sDo.setExecuteEndTime(LocalDateTime.now());
            Duration duration = Duration.between(sDo.getExecuteStartTime(), sDo.getExecuteEndTime());
            sDo.setExecuteSqlDelay(duration.toMillis());
            sqlExecuteRepo.save(sDo);

            // 预警参数
            Map<String, Object> alertParams = new HashMap<>(8);
            alertParams.put("success", exp == null ? "正常" : "异常");
            alertParams.put("failMsg", exp == null ? "/" : exp.getMessage());
            alertParams.put("rowNum", rowNums);
            alertParams.put("result", (rowNums > 0 ?  "失败" : "成功") + "(" + rowNums + "行)");
            alertParams.put("operateType", "查询");
            alertParams.put("sql", sDo.getQuerySql());
            alertParams.put("sqlName", sDo.getSqlName());
            alertParams.put("sqlCode", sDo.getSqlCode());
            alertParams.put("category", sDo.getCategory());
            alertParams.put("dsName", sDo.getDatasourceName());
//            alertParams.put("startTime", DatetimeUtil.toStr(sDo.getExecuteStartTime()));
//            alertParams.put("finishTime", DatetimeUtil.toStr(sDo.getExecuteEndTime()));
            alertParams.put("costTime", ObjectUtil.defaultIfNull(sDo.getExecuteSqlDelay(), 0));
            CompletableFuture.supplyAsync(() -> alertService.sendAlertByTmpl(SysAlertService.BUSINESS_TYPE_SQL_EXECUTE,
                            null, alertParams), taskExecutor)
                    .whenComplete((v, e) -> {
                        if (e == null) {
                            log.info("发送预警成功：{}", JSONUtil.toJsonString(v));
                            return;
                        }
                        log.info("发送预警失败：", e);
                    });

            return ApiResult.ok(sDo);
        } else {
            return ApiResult.fail("id不存在");
        }
    }

}
