package com.elitescloud.cloudt.platform.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;

import com.elitescloud.boot.log.common.OperationTypeEnum;
import com.elitescloud.boot.log.model.bo.OperationLogDTO;
import com.elitescloud.boot.log.service.OperationLogMqMessageService;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.platform.convert.UdcConvert;
import com.elitescloud.cloudt.platform.model.entity.QSysPlatformUdcDO;
import com.elitescloud.cloudt.platform.model.entity.SysPlatformUdcValueDO;
import com.elitescloud.cloudt.platform.model.params.udc.*;
import com.elitescloud.cloudt.platform.model.vo.SysPlatformUdcVO;
import com.elitescloud.cloudt.platform.model.vo.SysPlatformUdcValueMngVO;
import com.elitescloud.cloudt.platform.model.vo.SysPlatformUdcValueVO;
import com.elitescloud.cloudt.platform.service.SysPlatformUdcService;

import com.elitescloud.cloudt.platform.service.repo.SysPlatformAppRepo;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformUdcRepo;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformUdcValueRepo;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformUdcValueRepoProc;
import com.elitescloud.cloudt.system.service.SysTenantBasicDataService;
import com.elitescloud.cloudt.system.service.common.constant.BusinessObjectEnum;
import com.elitescloud.cloudt.system.service.config.TenantProperties;
import com.elitescloud.cloudt.system.service.util.JpaPredicateBuilder;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.core.task.TaskExecutor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * @Description:
 * @author: niu.chen
 * @date: 2022.09.15
 **/
@Service
@Slf4j
@RequiredArgsConstructor
public class SysPlatformUdcServiceImpl extends BaseServiceImpl implements SysPlatformUdcService {
    private final SysPlatformUdcRepo sysPlatformUdcRepo;
    private final SysPlatformUdcValueRepoProc sysPlatformUdcValueRepoProc;
    private final SysPlatformUdcValueRepo sysPlatformUdcValueRepo;
    private final SysPlatformAppRepo sysPlatformAppRepo;
    private final UdcProvider udcProvider;
    private final SysTenantBasicDataService tenantBasicDataService;
    @Autowired(required = false)
    private TenantProperties tenantProperties;

    private final OperationLogMqMessageService operationLogMqMessageService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> addUdc(AddUdcParam addUdcParam) {
        var appList = sysPlatformAppRepo.findByAppCode(addUdcParam.getAppCode());
        if (appList.size() != 1) {
            return ApiResult.fail(appList.isEmpty() ? "应用编码不存在" : "应用编码异常，存在多个");
        }

        var count = sysPlatformUdcRepo.countAllByAppCodeAndUdcCode(addUdcParam.getAppCode(), addUdcParam.getUdcCode());
        if (count != 0) {
            return ApiResult.fail("应用UDC编码重复");
        }
        var udcDo = UdcConvert.INSTANCE.saveParamToDo(addUdcParam);
        sysPlatformUdcRepo.saveAndFlush(udcDo);

        /***人工记录业务操作日志****/
        OperationLogDTO dto = operationLogMqMessageService.quickNewOperationLogDTO(
                BusinessObjectEnum.SysPlatformUdc,
                addUdcParam.getUdcCode(), OperationTypeEnum.ADD,
                "新增UDC");
        operationLogMqMessageService.sendAsyncOperationLogMqMessage(dto);

        return ApiResult.ok(udcDo.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> addUdcValue(AddUdcValueParam addUdcValueParam) {
        var udcDo = sysPlatformUdcRepo.findAllByAppCodeAndUdcCode(addUdcValueParam.getAppCode(),
                addUdcValueParam.getUdcCode());
        if (udcDo == null) {
            return ApiResult.fail("应用APP的UDC编码不存在," + addUdcValueParam.getAppCode() + "," + addUdcValueParam.getUdcCode());
        }

        var valueList = sysPlatformUdcValueRepoProc.queryValue(udcDo.getAppCode(), udcDo.getUdcCode(), null);
        if (!valueList.isEmpty()) {
            Set<String> codes = new HashSet<>();
            Set<String> names = new HashSet<>();
            for (SysPlatformUdcValueDO valueDO : valueList) {
                if (codes.contains(addUdcValueParam.getUdcValueCode())) {
                    return ApiResult.fail("值编码已存在");
                }
                codes.add(valueDO.getUdcValueCode());
                if (names.contains(addUdcValueParam.getUdcValueName())) {
                    return ApiResult.fail("值名称已存在");
                }
                names.add(valueDO.getUdcValueName());
            }
        }

        var udcValueDo = UdcConvert.INSTANCE.saveParamToDo(addUdcValueParam);
        udcValueDo.setAllowDefault(true);
        udcValueDo = sysPlatformUdcValueRepo.save(udcValueDo);
        // 清掉缓存
        this.clearCache(addUdcValueParam.getAppCode(), udcValueDo.getUdcCode());
        // 同步租户
        syncTenantUdc(udcDo.getAppCode(), udcDo.getUdcCode());

        /***人工记录业务操作日志****/
        OperationLogDTO dto = operationLogMqMessageService.quickNewOperationLogDTO(
                BusinessObjectEnum.SysPlatformUdc,
                udcValueDo.getUdcCode(), OperationTypeEnum.ADD,
                "新增UDC值" + udcValueDo.getUdcValueCode() + ":" + udcValueDo.getUdcValueName());
        operationLogMqMessageService.sendAsyncOperationLogMqMessage(dto);
        return ApiResult.ok(udcValueDo.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateUdcValue(UdcValueSaveVO saveVO) {
        Assert.notNull(saveVO.getId(), "ID为空");
        var udcValueDO = sysPlatformUdcValueRepoProc.get(saveVO.getId());
        Assert.notNull(udcValueDO, "修改的数据不存在");

        var valueList = sysPlatformUdcValueRepoProc.queryValue(udcValueDO.getAppCode(), udcValueDO.getUdcCode(), null);
        if (!valueList.isEmpty()) {
            Set<String> codes = new HashSet<>();
            Set<String> names = new HashSet<>();
            for (SysPlatformUdcValueDO valueDO : valueList) {
                if (valueDO.getId().equals(saveVO.getId())) {
                    continue;
                }
                if (codes.contains(saveVO.getUdcValueCode())) {
                    return ApiResult.fail("值编码已存在");
                }
                codes.add(valueDO.getUdcValueCode());
                if (names.contains(saveVO.getUdcValueName())) {
                    return ApiResult.fail("值名称已存在");
                }
                names.add(valueDO.getUdcValueName());
            }
        }
        //业务操作对象
        OperationLogDTO dto = operationLogMqMessageService.quickNewOperationLogDTO(
                BusinessObjectEnum.SysPlatformUdc, udcValueDO.getUdcCode(),
                OperationTypeEnum.UPDATE,
                "修改UDC值:" + udcValueDO.getUdcValueCode());


        //业务操作修改前数据
        dto.setOperationBeforeData(JSONUtil.toJsonString(udcValueDO));


        saveVO.setUdcOrder(ObjectUtil.defaultIfNull(saveVO.getUdcOrder(), 0));
        saveVO.setAllowStart(ObjectUtil.defaultIfNull(saveVO.getAllowStart(), true));
        UdcConvert.INSTANCE.copySaveVo2Do(saveVO, udcValueDO);
        sysPlatformUdcValueRepo.save(udcValueDO);

        //业务操作修改后数据
        dto.setOperationAfterData(JSONUtil.toJsonString(udcValueDO));


        // 同步租户
        syncTenantUdc(udcValueDO.getAppCode(), udcValueDO.getUdcCode());
        //发送业务操作日志
        operationLogMqMessageService.sendAsyncOperationLogMqMessage(dto);
        return ApiResult.ok(saveVO.getId());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Boolean> deleteFlagUdc(Long id) {
        sysPlatformUdcRepo.findById(id).ifPresentOrElse(
                sysPlatformUdcDO -> {
                    sysPlatformUdcDO.setDeleteFlag(1);
                    sysPlatformUdcRepo.save(sysPlatformUdcDO);
                    sysPlatformUdcValueRepo.findAllByAppCodeAndUdcCode(sysPlatformUdcDO.getAppCode(),
                                    sysPlatformUdcDO.getUdcCode())
                            .forEach(sysPlatformUdcValueDO -> {
                                sysPlatformUdcValueDO.setDeleteFlag(1);
                                sysPlatformUdcValueRepo.save(sysPlatformUdcValueDO);
                            });
                    // 清掉缓存
                    this.clearCache(sysPlatformUdcDO.getAppCode(), sysPlatformUdcDO.getUdcCode());
                    // 同步租户
                    syncTenantUdc(sysPlatformUdcDO.getAppCode(), sysPlatformUdcDO.getUdcCode());
                },
                () -> {
                    throw new BusinessException("id不存在");
                }
        );

        return ApiResult.ok(true);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Boolean> deleteFlagUdcValue(Long id) {
        sysPlatformUdcValueRepo.findById(id).ifPresentOrElse(sysPlatformUdcValueDO -> {
                    sysPlatformUdcValueDO.setDeleteFlag(1);
                    sysPlatformUdcValueRepo.save(sysPlatformUdcValueDO);
                    /***人工记录业务操作日志****/
                    OperationLogDTO dto = operationLogMqMessageService.quickNewOperationLogDTO(
                            BusinessObjectEnum.SysPlatformUdc,
                            sysPlatformUdcValueDO.getUdcCode(), OperationTypeEnum.DELETE,
                            "删除UDC值" + sysPlatformUdcValueDO.getUdcValueCode() + ":" + sysPlatformUdcValueDO.getUdcValueName());
                    operationLogMqMessageService.sendAsyncOperationLogMqMessage(dto);
                    // 清掉缓存
                    this.clearCache(sysPlatformUdcValueDO.getAppCode(), sysPlatformUdcValueDO.getUdcCode());

                    // 同步租户
                    syncTenantUdc(sysPlatformUdcValueDO.getAppCode(), sysPlatformUdcValueDO.getUdcCode());
                }
                , () -> {
                    throw new BusinessException("id不存在");
                });
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> deleteUdc(Long id) {
        var udcDoOp = sysPlatformUdcRepo.findById(id);
        if (udcDoOp.isPresent()) {
            var udcDo = udcDoOp.get();
            var list = sysPlatformUdcValueRepo.findAllByAppCodeAndUdcCode(udcDo.getAppCode(), udcDo.getUdcCode());
            if (list.size() > 0) {
                return ApiResult.fail("请先删除全部UDC子项");
            } else {
                sysPlatformUdcRepo.deleteById(id);
                /***人工记录业务操作日志****/
                OperationLogDTO dto = operationLogMqMessageService.quickNewOperationLogDTO(
                        BusinessObjectEnum.SysPlatformUdc,
                        udcDo.getUdcCode(), OperationTypeEnum.DELETE,
                        "删除UDC" + udcDo.getUdcCode() + ":" + udcDo.getUdcName());
                operationLogMqMessageService.sendAsyncOperationLogMqMessage(dto);
                this.clearCache(udcDo.getAppCode(), udcDo.getUdcCode());
                // 同步租户
                syncTenantUdc(udcDo.getAppCode(), udcDo.getUdcCode());
            }
        }

        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> deleteUdcValue(Long id) {
        var valueDO = sysPlatformUdcValueRepoProc.get(id);
        if (valueDO == null) {
            return ApiResult.noData();
        }
        sysPlatformUdcValueRepo.deleteById(id);
        /***人工记录业务操作日志****/
        OperationLogDTO dto = operationLogMqMessageService.quickNewOperationLogDTO(
                BusinessObjectEnum.SysPlatformUdc,
                valueDO.getUdcCode(), OperationTypeEnum.DELETE,
                "删除UDC值" + valueDO.getUdcValueCode() + ":" + valueDO.getUdcValueName());
        operationLogMqMessageService.sendAsyncOperationLogMqMessage(dto);
        // 清掉缓存
        this.clearCache(valueDO.getAppCode(), valueDO.getUdcCode());

        // 同步租户
        syncTenantUdc(valueDO.getAppCode(), valueDO.getUdcCode());
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> updateUdc(Long id, UpdateUdcParam updateUdcParam) {
        sysPlatformUdcRepo.findById(id).ifPresentOrElse(
                sysPlatformUdcDO -> {
                    //业务操作对象
                    OperationLogDTO dto = operationLogMqMessageService.quickNewOperationLogDTO(
                            BusinessObjectEnum.SysPlatformUdc, sysPlatformUdcDO.getUdcCode(),
                            OperationTypeEnum.UPDATE,
                            "修改UDC");
                    dto.setOperationBeforeData(JSONUtil.toJsonString(sysPlatformUdcDO));
//                    sysPlatformUdcDO.setAppCode(updateUdcParam.getAppCode());
//                    sysPlatformUdcDO.setUdcCode(updateUdcParam.getUdcCode());
                    sysPlatformUdcDO.setUdcName(updateUdcParam.getUdcName());
                    sysPlatformUdcDO.setAllowUpdate(updateUdcParam.getAllowUpdate());
                    sysPlatformUdcDO.setAllowAddValue(updateUdcParam.getAllowAddValue());
                    sysPlatformUdcDO.setUdcDescribe(updateUdcParam.getUdcDescribe());
                    sysPlatformUdcDO.setParentUdcCode(updateUdcParam.getParentUdcCode());
                    sysPlatformUdcRepo.save(sysPlatformUdcDO);
                    dto.setOperationAfterData(JSONUtil.toJsonString(sysPlatformUdcDO));
                    // 清掉缓存
                    this.clearCache(sysPlatformUdcDO.getAppCode(), sysPlatformUdcDO.getUdcCode());
                    // 同步租户
                    syncTenantUdc(sysPlatformUdcDO.getAppCode(), sysPlatformUdcDO.getUdcCode());
                    //发送业务操作日志
                    operationLogMqMessageService.sendAsyncOperationLogMqMessage(dto);
                }
                , () -> {
                    throw new BusinessException("id不存在");
                });
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> updateUdcValueAllowStart(Long id, Boolean allowStart) {
        sysPlatformUdcValueRepo.findById(id).ifPresentOrElse(
                sysPlatformUdcDO -> {
                    sysPlatformUdcDO.setAllowStart(allowStart);
                    sysPlatformUdcValueRepo.save(sysPlatformUdcDO);
                    // 清掉缓存
                    this.clearCache(sysPlatformUdcDO.getAppCode(), sysPlatformUdcDO.getUdcCode());
                    // 同步租户
                    syncTenantUdc(sysPlatformUdcDO.getAppCode(), sysPlatformUdcDO.getUdcCode());
                    /***人工记录业务操作日志****/
                    OperationLogDTO dto = operationLogMqMessageService.quickNewOperationLogDTO(
                            BusinessObjectEnum.SysPlatformUdc,
                            sysPlatformUdcDO.getUdcCode(), OperationTypeEnum.UPDATE,
                            "更新UDC状态：" + allowStart);
                    operationLogMqMessageService.sendAsyncOperationLogMqMessage(dto);

                }
                , () -> {
                    throw new BusinessException("id不存在");
                });
        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<SysPlatformUdcVO> getUdc(Long id) {
        AtomicReference<SysPlatformUdcVO> atomicUdcVO = new AtomicReference<SysPlatformUdcVO>();
        sysPlatformUdcRepo.findById(id).ifPresentOrElse(
                sysPlatformUdcDO -> {
                    SysPlatformUdcVO sysPlatformUdcVO1 = UdcConvert.INSTANCE.selectDOToVO(sysPlatformUdcDO);
                    var udcValueVOList = sysPlatformUdcValueRepo
                            .findAllByAppCodeAndUdcCode(sysPlatformUdcDO.getAppCode(),
                                    sysPlatformUdcDO.getUdcCode()).stream()
                            .map(UdcConvert.INSTANCE::selectDOToVO)
                            .sorted(comparatorOfUdcValue())
                            .collect(Collectors.toList());
                    sysPlatformUdcVO1.setSysPlatformUdcValueVOList(udcValueVOList);
                    atomicUdcVO.set(sysPlatformUdcVO1);
                }
                , () -> {
                    throw new BusinessException("id不存在");
                });
        return ApiResult.ok(atomicUdcVO.get());
    }

    @Override
    public ApiResult<PagingVO<SysPlatformUdcVO>> queryUdc(QueryUdcParam param) {
        var qdo = QSysPlatformUdcDO.sysPlatformUdcDO;
        var predicate = JpaPredicateBuilder.builder()
                .and(qdo.appCode::eq, param.getAppCode())
                .and(qdo.udcName::like, StringUtils.hasText(param.getUdcName()) ? "%" + param.getUdcName() + "%" : null)
                .and(qdo.udcCode::like, StringUtils.hasText(param.getUdcCode()) ? "%" + param.getUdcCode() + "%" : null)
                .getPredicate();
        var pageRequest = param.getPageRequest();
        if (pageRequest.getSort().isEmpty()) {
            pageRequest = pageRequest.withSort(Sort.by(Sort.Direction.DESC, "createTime"));
        }
        var page = sysPlatformUdcRepo.findAll(predicate, pageRequest);

        var pagingVo = PagingVO.<SysPlatformUdcVO>builder()
                .total(page.getTotalElements())
                .setRecords(page.get().map(UdcConvert.INSTANCE::selectDOToVO)
                        .collect(Collectors.toList()));
        return ApiResult.ok(pagingVo);

    }

    @Override
    public ApiResult<List<CodeNameParam>> getValueList(String appCode, String udcCode) {
        Assert.hasText(appCode, "应用编码为空");
        Assert.hasText(udcCode, "UDC编码为空");

        var result = sysPlatformUdcValueRepoProc.getValueList(appCode, udcCode, true);
        return ApiResult.ok(result);
    }

    @Override
    public List<SysPlatformUdcValueMngVO> queryUdcValue(HashMap<String,Object> param) {
        Assert.notNull(param.get("appCode").toString(), "应用编码不能为空！");
        Assert.notNull(param.get("udcCode").toString(), "udc编码不能为空！");
        List<SysPlatformUdcValueMngVO> udcValueMngVOList = sysPlatformUdcValueRepoProc.queryUdcValueList(param);
        if (CollectionUtils.isEmpty(udcValueMngVOList)){
            return null;
        }
        return udcValueMngVOList;
    }

    private void syncTenantUdc(String appCode, String udcCode) {
        if (tenantProperties == null || Boolean.FALSE.equals(tenantProperties.isAutoSyncBasicData())) {
            return;
        }
        taskExecutor.execute(() -> {
            try {
                TimeUnit.SECONDS.sleep(2);
            } catch (InterruptedException ignored) {
            }
            tenantBasicDataService.syncUdc(appCode, udcCode);
        });
    }

    private void clearCache(String appCode, String udcCode) {
        udcProvider.clearCache(appCode, udcCode);
    }

    private Comparator<SysPlatformUdcValueVO> comparatorOfUdcValue() {
        return Comparator.comparing(SysPlatformUdcValueVO::getUdcOrder, Comparator.nullsFirst(Integer::compareTo)).reversed();
    }
}
