package com.elitescloud.cloudt.platform.controller.api;

import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.log.common.BusinessObject;
import com.elitescloud.boot.log.common.BusinessObjectOperation;
import com.elitescloud.boot.log.common.OperationTypeEnum;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.constant.NextValuePeriod;
import com.elitescloud.cloudt.platform.model.params.seq.SysPlatformSeqNumPageQueryParam;
import com.elitescloud.cloudt.platform.model.params.seq.SysPlatformSeqNumSaveParam;
import com.elitescloud.cloudt.platform.model.params.seq.SysPlatformSeqNumUpdateParam;
import com.elitescloud.cloudt.platform.model.vo.sequence.SysPlatformSeqNumPageRespVO;
import com.elitescloud.cloudt.platform.model.vo.sequence.SysPlatformSeqNumRespVO;
import com.elitescloud.cloudt.platform.service.SysPlatformSeqNumMngService;
import com.elitescloud.cloudt.system.service.common.constant.BusinessObjectConstant;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 下一编号.
 *
 * @author niuchen
 * 2022/9/19
 */
@Api(tags = "发号器-下一编号管理")
@RestController
@RequestMapping(value = "/sys/platform/seq/num", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
@BusinessObject(businessType = BusinessObjectConstant.SYS_PLATFORM_NEXT_NUMBER)
public class SysPlatformSeqNumMngController {

    public final SysPlatformSeqNumMngService sysPlatformSeqNumMngService;

    public SysPlatformSeqNumMngController(SysPlatformSeqNumMngService sysPlatformSeqNumMngService) {
        this.sysPlatformSeqNumMngService = sysPlatformSeqNumMngService;
    }

    /**
     * 分页查询记录
     *
     * @param queryVO 查询参数
     * @return 分页数据
     */
    @ApiOperation(value = "分页查询记录")
    @ApiOperationSupport(order = 1)
    @PostMapping(value = "/page")
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.SELECT_PAGE,operationDescription = "分页查询记录"
    )
    public ApiResult<PagingVO<SysPlatformSeqNumPageRespVO>> page(@RequestBody SysPlatformSeqNumPageQueryParam queryVO) {
        return sysPlatformSeqNumMngService.page(queryVO);
    }

    /**
     * 切换启用禁用
     *
     * @param id 记录ID
     * @return 切换结果
     */
    @ApiOperation(value = "切换启用禁用")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParam(name = "id", value = "记录ID", required = true, dataType = "int")
    @PatchMapping(value = "/switch/enabled/{id}/{enabled}")
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.UPDATE,operationDescription = "切换启用禁用",
            logEnabled = true,  argsJsonEnabled = true,resultJsonEnabled = true
    )
    public ApiResult<Long> switchEnabled(@PathVariable("id") Long id,@PathVariable("enabled") Boolean enabled) {
        return sysPlatformSeqNumMngService.switchEnabled(id,enabled);
    }

    /**
     * 删除记录
     *
     * @param id 记录ID
     * @return 删除结果
     */
    @ApiOperation(value = "删除记录")
    @ApiOperationSupport(order = 3)
    @ApiImplicitParam(name = "id", value = "记录ID", required = true, dataType = "int")
    @DeleteMapping(value = "/del/{id}")
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.DELETE,operationDescription = "删除记录",
            logEnabled = true,  argsJsonEnabled = true,resultJsonEnabled = true
    )
    public ApiResult<Long> delete(@PathVariable("id") Long id) {

        return sysPlatformSeqNumMngService.delete(id);
    }

    /**
     * 获取记录详情
     *
     * @param id 记录ID
     * @return 记录信息
     */
    @ApiOperation(value = "获取记录详情")
    @ApiOperationSupport(order = 4)
    @ApiImplicitParam(name = "id", value = "记录ID", required = true, dataType = "int")
    @GetMapping(value = "/get/{id}")
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.SELECT,operationDescription = "获取记录详情"
    )
    public ApiResult<SysPlatformSeqNumRespVO> get(@PathVariable("id") Long id) {

        return sysPlatformSeqNumMngService.get(id);
    }

    /**
     * 新增记录
     *
     * @param saveVO 记录信息
     * @return 新增结果
     */
    @ApiOperation(value = "新增记录")
    @ApiOperationSupport(order = 5)
    @PostMapping(value = "/save")
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.ADD,operationDescription = "新增记录",
            logEnabled = true,  argsJsonEnabled = true,resultJsonEnabled = true
    )
    public ApiResult<Long> save(@RequestBody @Valid SysPlatformSeqNumSaveParam saveVO) {
        return sysPlatformSeqNumMngService.save(saveVO);
    }

    /**
     * 修改记录
     *
     * @param id     记录ID
     * @param saveVO 记录信息
     * @return 修改结果
     */
    @ApiOperation(value = "修改记录")
    @ApiOperationSupport(order = 6)
    @ApiImplicitParam(name = "id", value = "记录ID", required = true, dataType = "int")
    @PutMapping(value = "/update/{id}")
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.UPDATE,operationDescription = "修改记录",
            logEnabled = true,  argsJsonEnabled = true,resultJsonEnabled = true
    )
    public ApiResult<Long> update(@PathVariable("id") Long id, @RequestBody @Valid SysPlatformSeqNumUpdateParam saveVO) {
        return sysPlatformSeqNumMngService.update(id,saveVO);
    }

    /**
     * 编号周期
     *
     * @return 编号周期列表
     */
    @ApiOperation(value = "编号周期")
    @ApiOperationSupport(order = 7)
    @GetMapping(value = "/periods")
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.SELECT,operationDescription = "编号周期"
    )
    public ApiResult<List<CodeNameParam>> numPeriods() {
        var values = Arrays.stream(NextValuePeriod.values())
                .map(t -> new CodeNameParam(t.name(), t.getDescription()))
                .collect(Collectors.toList());
        return ApiResult.ok(values);
    }
}
