package com.elitescloud.cloudt.platform.controller.api.extend;

import com.elitescloud.boot.log.common.BusinessObject;
import com.elitescloud.boot.log.common.BusinessObjectOperation;
import com.elitescloud.boot.log.common.OperationTypeEnum;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.platform.model.vo.extend.query.CurrencyPageQueryVO;
import com.elitescloud.cloudt.platform.model.vo.extend.resp.CurrencyDetailRespVO;
import com.elitescloud.cloudt.platform.model.vo.extend.resp.CurrencyPageRespVO;
import com.elitescloud.cloudt.platform.model.vo.extend.save.CurrencySaveVO;
import com.elitescloud.cloudt.platform.service.SysPlatformCurrencyService;
import com.elitescloud.cloudt.system.service.common.constant.BusinessObjectConstant;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;

/**
 * 货币管理.
 *
 * @author Kaiser（wang shao）
 * 2023/1/13
 */
@Api(tags = "货币管理")
@RestController
@RequestMapping(value = "/platform/mng/currency", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
@BusinessObject(businessType = BusinessObjectConstant.SYS_PLATFORM_CURRENCY)
public class SysPlatformCurrencyController {

    private final SysPlatformCurrencyService service;

    public SysPlatformCurrencyController(SysPlatformCurrencyService service) {
        this.service = service;
    }

    /**
     * 分页查询货币
     *
     * @param queryVO 查询参数
     * @return 货币列表
     */
    @PostMapping(value = "/q")
    @ApiOperation(value = "分页查询货币")
    @ApiOperationSupport(order = 1)
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.SELECT_PAGE,operationDescription = "查询货币"
    )
    public ApiResult<PagingVO<CurrencyPageRespVO>> page(@RequestBody CurrencyPageQueryVO queryVO) {
        return service.page(queryVO);
    }


    /**
     * 切换启用禁用
     *
     * @param id 货币ID
     * @return 切换结果
     */
    @PatchMapping(value = "/switch/{id}/enabled")
    @ApiOperation(value = "切换启用禁用")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParam(name = "id", value = "货币ID", required = true)
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.UPDATE,operationDescription = "启用禁用"
            , logEnabled = true,  argsJsonEnabled = true,resultJsonEnabled = true
    )
    public ApiResult<Long> switchEnabled(@PathVariable("id") Long id) {
        return service.switchEnabled(id);
    }

    /**
     * 获取货币详情
     *
     * @param id 货币ID
     * @return 货币详情
     */
    @GetMapping(value = "/{id}")
    @ApiOperation(value = "获取货币详情")
    @ApiOperationSupport(order = 3)
    @ApiImplicitParam(name = "id", value = "货币ID", required = true)
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.SELECT,operationDescription = "货币详情"
    )
    public ApiResult<CurrencyDetailRespVO> get(@PathVariable("id") Long id) {
        return service.get(id);
    }

    /**
     * 保存货币
     *
     * @param saveVO 货币信息
     * @return 货币ID
     */
    @PostMapping(value = "/save")
    @ApiOperation(value = "保存货币")
    @ApiOperationSupport(order = 4)
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.SAVE,operationDescription = "保存货币"
            , logEnabled = true,  argsJsonEnabled = true,resultJsonEnabled = true
    )
    public ApiResult<Long> save(@RequestBody @Valid CurrencySaveVO saveVO) {
        return service.save(saveVO);
    }

    /**
     * 删除货币信息
     *
     * @param id 货币ID
     * @return 货币ID
     */
    @DeleteMapping(value = "/{id}")
    @ApiOperation(value = "删除货币信息")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParam(name = "id", value = "货币ID", required = true)
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.DELETE,operationDescription = "删除货币"
            , logEnabled = true,  argsJsonEnabled = true,resultJsonEnabled = true
    )
    public ApiResult<Long> delete(@PathVariable("id") Long id) {
        return service.delete(id);
    }
}
