package com.elitescloud.cloudt.system.provider.org;

import com.elitescloud.cloudt.Application;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.dto.SysEmployeeBasicDTO;
import com.elitescloud.cloudt.system.dto.SysEmployeeDetailDTO;
import com.elitescloud.cloudt.system.dto.req.EmployeePageQueryDTO;
import com.elitescloud.cloudt.system.dto.req.EmployeeQueryDTO;
import com.elitescloud.cloudt.system.dto.req.EmployeeUpsertDTO;
import com.elitescloud.cloudt.system.dto.resp.EmployeePageRespDTO;
import com.elitescloud.cloudt.system.dto.resp.EmployeeUnderlingDTO;
import com.elitesland.cbpl.unicom.annotation.Unicom;
import com.elitesland.cbpl.unicom.domain.InvokeDomain;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Set;

/**
 * 员工相关服务.
 *
 * @author Kaiser（wang shao）
 * 2022/12/20
 */
@FeignClient(name = Application.NAME, path = EmployeeRpcService.URI)
@Validated
@Unicom(domain = InvokeDomain.SYSTEM)
public interface EmployeeRpcService {
    String URI = Application.URI_PREFIX + "/employee";

    /**
     * 员工创建与修改
     * <p>
     * 根据id或code确定员工唯一性，如果存在则修改，否则新增
     *
     * @param upsertDTO 员工信息
     * @return 员工编号
     */
    @PostMapping(value = "/upsert")
    ApiResult<String> upsert(@RequestBody @Valid EmployeeUpsertDTO upsertDTO);

    /**
     * 根据员工编号修改员工状态
     *
     * @param code    员工编号
     * @param enabled 启用状态
     * @return 员工编号
     */
    @PatchMapping(value = "/updateEnabledByCode")
    ApiResult<String> updateEnabledByCode(@NotBlank(message = "员工编号为空") @RequestParam("code") String code,
                                          @NotNull(message = "启用状态为空") @RequestParam("enabled") Boolean enabled);

    /**
     * 根据员工ID获取员工信息
     *
     * @param id 员工ID
     * @return 员工信息
     */
    @GetMapping(value = "/getById")
    ApiResult<SysEmployeeBasicDTO> getById(@NotNull(message = "员工ID为空") @RequestParam(name = "id") Long id);

    /**
     * 根据员工ID获取员工信息
     *
     * @param code 员工编码
     * @return 员工信息
     */
    @GetMapping(value = "/getByCode")
    ApiResult<SysEmployeeBasicDTO> getByCode(@NotBlank(message = "员工编号为空") @RequestParam(name = "code") String code);

    /**
     * 根据账号获取员工信息
     *
     * @param username 登录账号
     * @return 员工信息
     */
    @GetMapping(value = "/getByUsername")
    ApiResult<SysEmployeeBasicDTO> getByUsername(@NotBlank(message = "用户账号为空") @RequestParam(name = "username") String username);

    /**
     * 根据用户ID获取所属员工
     *
     * @param userId 用户ID
     * @return 员工信息
     */
    @GetMapping(value = "/getByUserId")
    ApiResult<SysEmployeeBasicDTO> getByUserId(@NotNull(message = "用户ID为空") @RequestParam(name = "id") Long userId);

    /**
     * 根据员工编号获取详细信息
     *
     * @param code 员工编号
     * @return 详细信息
     */
    @GetMapping(value = "/getDetailByCode")
    ApiResult<SysEmployeeDetailDTO> getDetailByCode(@NotBlank(message = "员工编号为空") @RequestParam(name = "code") String code);

    /**
     * 根据员工编号获取详细信息
     *
     * @param codes 员工编号
     * @return 详细信息
     */
    @PostMapping(value = "/getDetailListByCode")
    ApiResult<List<SysEmployeeDetailDTO>> getDetailListByCode(@RequestBody @NotEmpty(message = "员工编号为空") Set<String> codes);

    /**
     * 获取下属员工
     *
     * @param code    员工编号
     * @param cascade 是否级联获取所有下属员工
     * @return 员工列表
     */
    @GetMapping(value = "/getUnderlingByCode")
    ApiResult<List<EmployeeUnderlingDTO>> getUnderlingByCode(@NotBlank(message = "员工编号为空") @RequestParam(name = "code") String code,
                                                             @RequestParam(name = "cascade", required = false) Boolean cascade);

    /**
     * 查询员工列表
     *
     * @param queryDTO 查询参数
     * @return 员工列表
     */
    @PostMapping(value = "/queryList")
    ApiResult<List<SysEmployeeBasicDTO>> queryList(@RequestBody EmployeeQueryDTO queryDTO);

    /**
     * 分页查询员工
     *
     * @param queryDTO 查询参数
     * @return 员工列表
     */
    @PostMapping(value = "/queryByPage")
    ApiResult<PagingVO<EmployeePageRespDTO>> queryByPage(@RequestBody EmployeePageQueryDTO queryDTO);
}
