package com.elitescloud.cloudt.system.provider.org;

import com.elitescloud.cloudt.Application;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.dto.SysUserBasicDTO;
import com.elitescloud.cloudt.system.dto.req.UserCreateDTO;
import com.elitescloud.cloudt.system.dto.req.UserQueryDTO;
import com.elitesland.cbpl.unicom.annotation.Unicom;
import com.elitesland.cbpl.unicom.domain.InvokeDomain;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * 用户账号接口.
 *
 * @author Kaiser（wang shao）
 * 2022/12/27
 */
@FeignClient(name = Application.NAME, path = UserRpcService.URI)
@Validated
@Unicom(domain = InvokeDomain.SYSTEM)
public interface UserRpcService {

    String URI = Application.URI_PREFIX + "/v3/user";

    /**
     * 新增、更新用户账号
     *
     * @param userCreateDTO 账号信息
     * @return 用户ID
     */
    @PostMapping(value = "/upsert")
    ApiResult<Long> upsert(@RequestBody @Valid UserCreateDTO userCreateDTO);

    /**
     * 更新用户密码
     *
     * @param id  用户ID
     * @param pwd 新的密码，为空则设置为默认密码
     * @return 用户ID
     */
    @PatchMapping(value = "/update/pwd")
    ApiResult<Long> updatePwd(@NotNull(message = "用户ID为空") @RequestParam(name = "id") Long id,
                              @RequestParam(name = "pwd", required = false) String pwd);

    /**
     * 更新用户启用状态
     *
     * @param id      用户ID
     * @param enabled 启用状态
     * @return 用户ID
     */
    @PatchMapping(value = "/update/enabled")
    ApiResult<Long> updateEnabled(@NotNull(message = "用户ID为空") @RequestParam(name = "id") Long id,
                                  @NotNull(message = "启用状态为空") @RequestParam(name = "enabled") Boolean enabled);

    /**
     * 删除用户
     *
     * @param id 用户ID
     * @return 用户ID
     */
    @DeleteMapping(value = "/delete")
    ApiResult<Long> delete(@NotNull(message = "用户ID为空") @RequestParam(name = "id") Long id);

    /**
     * 根据用户ID获取用户信息
     *
     * @param id 用户ID
     * @return 用户信息
     */
    @GetMapping(value = "/getById")
    ApiResult<SysUserBasicDTO> getById(@NotNull(message = "用户ID为空") @RequestParam(name = "id") Long id);

    /**
     * 根据用户ID获取用户信息
     *
     * @param ids 用户ID
     * @return 用户信息
     */
    @PostMapping(value = "/getByIds")
    ApiResult<List<SysUserBasicDTO>> getByIds(@RequestBody @NotEmpty(message = "用户ID为空") List<Long> ids);

    /**
     * 根据用户账号获取用户ID
     *
     * @param username 用户账号
     * @return 用户ID
     */
    @GetMapping(value = "/getIdByUsername")
    ApiResult<Long> getIdByUsername(@NotBlank(message = "用户账号为空") @RequestParam(name = "username") String username);

    /**
     * 判断用户账号是否存在
     *
     * @param username 用户账号
     * @return 是否存在
     */
    @GetMapping(value = "/exists/username")
    ApiResult<Boolean> existsUsername(@NotBlank(message = "用户账号为空") @RequestParam(name = "username") String username);

    /**
     * 判断手机号是否存在
     *
     * @param mobile 手机号
     * @return 是否存在
     */
    @GetMapping(value = "/exists/mobile")
    ApiResult<Boolean> existsMobile(@NotBlank(message = "手机号为空") @RequestParam(name = "mobile") String mobile);

    /**
     * 判断邮箱是否存在
     *
     * @param email 邮箱
     * @return 是否存在
     */
    @GetMapping(value = "/exists/email")
    ApiResult<Boolean> existsEmail(@NotBlank(message = "邮箱为空") @RequestParam(name = "email") String email);

    /**
     * 查询用户列表
     *
     * @param queryParam 查询参数
     * @return 用户列表
     */
    @PostMapping(value = "/query")
    ApiResult<List<SysUserBasicDTO>> queryUser(@NotNull @RequestBody UserQueryDTO queryParam);
}
