package com.elitescloud.cloudt.system.provider.org;

import com.elitescloud.cloudt.Application;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.dto.SysEmployeeBasicDTO;
import com.elitescloud.cloudt.system.dto.SysEmployeeDetailDTO;
import com.elitescloud.cloudt.system.dto.req.EmployeePageQueryDTO;
import com.elitescloud.cloudt.system.dto.req.EmployeeQueryDTO;
import com.elitescloud.cloudt.system.dto.req.EmployeeUpsertDTO;
import com.elitescloud.cloudt.system.dto.resp.EmployeePageRespDTO;
import com.elitescloud.cloudt.system.dto.resp.EmployeeUnderlingDTO;
import com.elitesland.cbpl.unicom.annotation.Unicom;
import com.elitesland.cbpl.unicom.domain.InvokeDomain;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Set;

/**
 * 员工相关服务.
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/20
 */
@Validated
@Unicom(domain = InvokeDomain.SYSTEM, path = EmployeeRpcService.URI)
public interface EmployeeRpcService {
    String URI = Application.URI_PREFIX + "/employee";

    /**
     * 员工创建与修改
     * <p>
     * 根据id或code确定员工唯一性，如果存在则修改，否则新增
     *
     * @param upsertDTO 员工信息
     * @return 员工编号
     */
    @PostMapping(value = "/upsert")
    ApiResult<String> upsert(@RequestBody @Valid EmployeeUpsertDTO upsertDTO);

    /**
     * 批量保存员工信息
     *
     * @param upsertDTOList 员工信息列表
     * @return 校验结果
     */
    @PostMapping(value = "/upsertBatch")
    ApiResult<List<String>> upsertBatch(@RequestBody @NotEmpty(message = "员工信息为空") @Valid List<EmployeeUpsertDTO> upsertDTOList);

    /**
     * 检查员工信息
     *
     * @param upsertDTOList 员工信息列表
     * @return 校验结果
     */
    @PostMapping(value = "/checkBatch")
    ApiResult<List<String>> checkBatch(@RequestBody @NotEmpty(message = "员工信息为空") @Valid List<EmployeeUpsertDTO> upsertDTOList);

    /**
     * 根据员工编号修改员工状态
     *
     * @param code    员工编号
     * @param enabled 启用状态
     * @return 员工编号
     */
    @PatchMapping(value = "/updateEnabledByCode")
    ApiResult<String> updateEnabledByCode(@NotBlank(message = "员工编号为空") @RequestParam("code") String code,
                                          @NotNull(message = "启用状态为空") @RequestParam("enabled") Boolean enabled);

    /**
     * 更新手机号
     *
     * @param id         用户ID
     * @param mobile     手机号
     * @param updateUser 是否更新账号里的手机号
     * @return 用户ID
     */
    @PutMapping(value = "/update/mobile")
    ApiResult<Long> updateMobile(@NotNull(message = "用户ID为空") @RequestParam("id") Long id,
                                 @RequestParam(name = "mobile", required = false) String mobile,
                                 @RequestParam(name = "updateUser", defaultValue = "false") Boolean updateUser);

    /**
     * 更新邮箱
     *
     * @param id         用户ID
     * @param email      邮箱
     * @param updateUser 是否更新账号里的邮箱
     * @return 用户ID
     */
    @PutMapping(value = "/update/email")
    ApiResult<Long> updateEmail(@NotNull(message = "用户ID为空") @RequestParam("id") Long id,
                                @RequestParam(name = "email", required = false) String email,
                                @RequestParam(name = "updateUser", defaultValue = "false") Boolean updateUser);

    /**
     * 根据员工ID获取员工信息
     *
     * @param id 员工ID
     * @return 员工信息
     */
    @GetMapping(value = "/getById")
    ApiResult<SysEmployeeBasicDTO> getById(@NotNull(message = "员工ID为空") @RequestParam(name = "id") Long id);

    /**
     * 根据员工ID获取员工信息
     *
     * @param code 员工编码
     * @return 员工信息
     */
    @GetMapping(value = "/getByCode")
    ApiResult<SysEmployeeBasicDTO> getByCode(@NotBlank(message = "员工编号为空") @RequestParam(name = "code") String code);

    /**
     * 根据账号获取员工信息
     *
     * @param username 登录账号
     * @return 员工信息
     */
    @GetMapping(value = "/getByUsername")
    ApiResult<SysEmployeeBasicDTO> getByUsername(@NotBlank(message = "用户账号为空") @RequestParam(name = "username") String username);

    /**
     * 根据用户ID获取所属员工
     *
     * @param userId 用户ID
     * @return 员工信息
     */
    @GetMapping(value = "/getByUserId")
    ApiResult<SysEmployeeBasicDTO> getByUserId(@NotNull(message = "用户ID为空") @RequestParam(name = "id") Long userId);

    /**
     * 根据员工编号获取详细信息
     *
     * @param code 员工编号
     * @return 详细信息
     */
    @GetMapping(value = "/getDetailByCode")
    ApiResult<SysEmployeeDetailDTO> getDetailByCode(@NotBlank(message = "员工编号为空") @RequestParam(name = "code") String code);

    /**
     * 根据员工编号获取详细信息
     *
     * @param codes 员工编号
     * @return 详细信息
     */
    @PostMapping(value = "/getDetailListByCode")
    ApiResult<List<SysEmployeeDetailDTO>> getDetailListByCode(@RequestBody @NotEmpty(message = "员工编号为空") Set<String> codes);

    /**
     * 获取下属员工
     *
     * @param code    员工编号
     * @param cascade 是否级联获取所有下属员工，默认否
     * @param enabled 员工状态
     * @return 员工列表
     */
    @GetMapping(value = "/getUnderlingByCode")
    ApiResult<List<EmployeeUnderlingDTO>> getUnderlingByCode(@NotBlank(message = "员工编号为空") @RequestParam(name = "code") String code,
                                                             @RequestParam(name = "cascade", required = false, defaultValue = "false") Boolean cascade,
                                                             @RequestParam(name = "enabled", required = false) Boolean enabled);

    /**
     * 查询员工列表
     *
     * @param queryDTO 查询参数
     * @return 员工列表
     */
    @PostMapping(value = "/queryList")
    ApiResult<List<SysEmployeeBasicDTO>> queryList(@RequestBody EmployeeQueryDTO queryDTO);

    /**
     * 分页查询员工
     *
     * @param queryDTO 查询参数
     * @return 员工列表
     */
    @PostMapping(value = "/queryByPage")
    ApiResult<PagingVO<EmployeePageRespDTO>> queryByPage(@RequestBody EmployeePageQueryDTO queryDTO);
}
