package com.elitescloud.cloudt.tenant.config;

import com.elitescloud.cloudt.common.condition.ConditionalOnRpc;
import com.elitescloud.cloudt.common.util.RedisUtils;
import com.elitescloud.cloudt.context.redis.RedisWrapper;
import com.elitescloud.cloudt.tenant.config.support.ServerTenantProvider;
import com.elitescloud.cloudt.tenant.controller.*;
import com.elitescloud.cloudt.tenant.controller.api.ApiTenantController;
import com.elitescloud.cloudt.tenant.provider.TenantProvider;
import com.elitescloud.cloudt.tenant.rpc.TenantRpcProvider;
import com.elitescloud.cloudt.tenant.rpc.TenantRpcProviderImpl;
import com.elitescloud.cloudt.tenant.service.*;
import com.elitescloud.cloudt.tenant.service.impl.*;
import com.elitescloud.cloudt.tenant.service.manager.SysTenantManager;
import com.elitescloud.cloudt.tenant.service.manager.TenantClientCaller;
import com.elitescloud.cloudt.tenant.service.manager.impl.DubboTenantClientCaller;
import com.elitescloud.cloudt.tenant.service.manager.impl.OpenFeignTenantClientCaller;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Primary;

/**
 * 租户配置.
 *
 * @author Kaiser（wang shao）
 * 2022/3/25
 */
@Configuration(proxyBeanMethods = false)
@ConditionalOnProperty(prefix = TenantClientProperties.CONFIG_PREFIX, name = "enabled", havingValue = "true", matchIfMissing = true)
@EnableConfigurationProperties(TenantProperties.class)
@Import({TenantConfig.SupportedService.class})
public class TenantConfig implements ApplicationRunner {

    private SysTenantBaseDataSyncService tenantBaseDataSyncService;

    @Bean
    @ConditionalOnRpc(requiredDubbo = true)
    public TenantClientCaller dubboTenantClientCaller() {
        return new DubboTenantClientCaller();
    }

    @Bean
    @ConditionalOnMissingBean(TenantClientCaller.class)
    @ConditionalOnRpc(requiredOpenFeign = true)
    public TenantClientCaller openFeignTenantClientCaller(ApplicationContext applicationContext) {
        return new OpenFeignTenantClientCaller(applicationContext);
    }

    @Bean
    @Primary
    public TenantProvider serverTenantProvider(TenantRpcProvider tenantRpcProvider,
                                               ObjectProvider<RedisUtils> redisUtils,
                                               ObjectProvider<RedisWrapper> redisWrapper,
                                               SysTenantQueryService tenantService) {
        return new ServerTenantProvider(tenantRpcProvider, redisUtils.getIfAvailable(), redisWrapper.getIfAvailable(), tenantService);
    }

    @Override
    public void run(ApplicationArguments args) throws Exception {
        if (tenantBaseDataSyncService != null) {
            tenantBaseDataSyncService.execDealUnFinished();
        }
    }

    @Autowired(required = false)
    public void setTenantBaseDataSyncService(SysTenantBaseDataSyncService tenantBaseDataSyncService) {
        this.tenantBaseDataSyncService = tenantBaseDataSyncService;
    }

    /**
     * 手动注册的服务
     */
    static class SupportedService {
        @Bean
        public ApiTenantController apiTenantController() {
            return new ApiTenantController();
        }

        @Bean
        public SysTenantAppController sysTenantAppController() {
            return new SysTenantAppController();
        }

        @Bean
        public SysTenantBaseDataSyncController sysTenantBaseDataSyncController(SysTenantBaseDataSyncService sysTenantBaseDataSyncService) {
            return new SysTenantBaseDataSyncController(sysTenantBaseDataSyncService);
        }

        @Bean
        public SysTenantManager sysTenantManager() {
            return new SysTenantManager();
        }

        @Bean
        public SysTenantBaseDataSyncService sysTenantBaseDataSyncService() {
            return new SysTenantBaseDataSyncServiceImpl();
        }

        @Bean
        public SysTenantController sysTenantController() {
            return new SysTenantController();
        }

        @Bean
        public SysTenantDbMigrateController sysTenantDbMigrateController() {
            return new SysTenantDbMigrateController();
        }

        @Bean
        public SysTenantUserController sysTenantUserController() {
            return new SysTenantUserController();
        }

        @Bean
        public TenantRpcProviderImpl tenantRpcProvider() {
            return new TenantRpcProviderImpl();
        }

        @Bean
        public ApiTenantService apiTenantService() {
            return new ApiTenantServiceImpl();
        }

        @Bean
        public SysTenantAppService sysTenantAppService() {
            return new SysTenantAppServiceImpl();
        }

        @Bean
        public SysTenantDbMigrateService sysTenantDbMigrateService() {
            return new SysTenantDbMigrateServiceImpl();
        }

        @Bean
        public SysTenantMngService sysTenantService() {
            return new SysTenantMngServiceImpl();
        }

        @Bean
        public SysTenantQueryService sysTenantQueryService() {
            return new SysTenantQueryServiceImpl();
        }
    }
}
