package com.elitescloud.cloudt.tenant.service.impl;

import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.context.threadpool.ThreadPoolAutoConfiguration;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitescloud.cloudt.tenant.model.entity.SysTenantBaseDataSyncDO;
import com.elitescloud.cloudt.tenant.service.SysTenantBaseDataSyncService;
import com.elitescloud.cloudt.tenant.service.manager.SysTenantManager;
import com.elitescloud.cloudt.tenant.service.repo.SysTenantBaseDataSyncRepo;
import com.elitescloud.cloudt.tenant.service.repo.SysTenantBaseDataSyncRepoProc;
import com.elitescloud.cloudt.tenant.service.repo.SysTenantRepoProc;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.core.task.TaskExecutor;
import org.springframework.transaction.annotation.Transactional;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.concurrent.CompletableFuture;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/12
 */
@Log4j2
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class SysTenantBaseDataSyncServiceImpl extends BaseServiceImpl implements SysTenantBaseDataSyncService {

    @Autowired
    private SysTenantRepoProc tenantRepoProc;
    @Autowired
    private SysTenantBaseDataSyncRepo baseDataSyncRepo;
    @Autowired
    private SysTenantBaseDataSyncRepoProc baseDataSyncRepoProc;

    @Autowired
    private SysTenantManager tenantManager;

    @Autowired
    @Qualifier(ThreadPoolAutoConfiguration.BEAN_NAME)
    private TaskExecutor taskExecutor;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> execSyncData(Long tenantId) {
        var tenant = tenantRepoProc.get(tenantId);
        if (tenant == null) {
            return ApiResult.fail("租户不存在");
        }

        // 判断目前是否有进行中的
        var record = baseDataSyncRepoProc.getLastOfUnFinished(tenantId);
        if (record != null && Duration.between(record.getStartTime(), LocalDateTime.now()).compareTo(Duration.ofMinutes(20)) < 0) {
            return ApiResult.fail("还有进行中的同步记录，请稍后后再操作");
        }

        // 发起同步
        addSync(tenantId);

        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> execDealUnFinished() {
        baseDataSyncRepoProc.updateFinishedAll();
        return ApiResult.ok(true);
    }

    private void addSync(Long tenantId) {
        // 先保存同步记录
        SysTenantBaseDataSyncDO syncDO = new SysTenantBaseDataSyncDO();
        syncDO.setSysTenantId(tenantId);
        syncDO.setFinished(false);
        syncDO.setSuccess(false);
        baseDataSyncRepo.save(syncDO);
        tenantRepoProc.updateBaseDataSyncId(tenantId, syncDO.getId());

        // 调用同步
        CompletableFuture.runAsync(() -> tenantManager.syncBaseData(syncDO.getId(), tenantId), taskExecutor).whenComplete((res, e) -> {
            tenantManager.updateSyncResult(syncDO.getId(), e);
        });
    }
}
