package com.elitescloud.cloudt.tenant.provider;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.cloudt.core.tenant.support.TenantClientProvider;
import com.elitescloud.cloudt.system.constant.TenantType;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitescloud.cloudt.system.spi.SysUserLoginSpi;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitescloud.cloudt.tenant.service.SysTenantQueryService;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.util.Collections;
import java.util.List;

/**
 * 用户登录信息的扩展.
 * <p>
 * 在当前登录用户信息中加入租户信息
 *
 * @author Kaiser（wang shao）
 * 2022/3/29
 */
@Log4j2
public class SysUserLoginTenantProvider implements SysUserLoginSpi {

    @Autowired(required = false)
    private SysTenantQueryService tenantService;
    @Autowired
    private TenantClientProvider tenantClientProvider;

    @Override
    public String getServiceName() {
        return "租户信息服务";
    }

    @Override
    public void expendLoginUserInfo(SysUserDTO user) {
        user.setSysTenantDTOList(Collections.emptyList());
        if (!tenantClientProvider.enabledTenant()) {
            // 未启用多租户
            return;
        }

        // 查询用户的所在租户
        var tenants = tenantService.queryTenantDtoOfUser(user.getId()).getData();
        user.setSysTenantDTOList(ObjectUtil.defaultIfNull(tenants, Collections.emptyList()));

        // 登录检查
        chooseCurrentTenant(user, tenants);
        log.info("租户：{}", tenants);
        log.info("当前租户：{}", user.getSysTenantVO());
    }

    private void chooseCurrentTenant(SysUserDTO user, List<SysTenantDTO> tenants) {
        if (CollectionUtils.isEmpty(tenants)) {
            return;
        }
        // 判断请求域名所属租户
        var requestTenant = tenantClientProvider.getCurrentTenant();

        // 匹配当前租户
        if (requestTenant != null) {
            var matchTenant = tenants.stream()
                    .filter(t -> t.getId().longValue() == requestTenant.getId())
                    .findAny();
            if (matchTenant.isPresent()) {
                user.setSysTenantVO(matchTenant.get());
                return;
            }

            // 取第一个租户作为默认租户
            user.setSysTenantVO(tenants.get(0));
            return;
        }

        // 取第一个租户作为默认租户
        user.setSysTenantVO(tenants.get(0));

        // 默认域名，优先运营机构
        if (tenantClientProvider.isDefaultDomainRequest()) {
            var operationTenant = tenants.stream().filter(t -> t.getType() == TenantType.OPERATION).findAny();
            operationTenant.ifPresent(user::setSysTenantVO);
        }
    }
}
