package com.elitescloud.cloudt.tenant.provider;

import com.elitescloud.cloudt.authorization.core.SecurityContextUtil;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitescloud.cloudt.core.tenant.support.TenantClientProvider;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.service.model.entity.QSysTenantUserDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysUserDO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserDO;
import com.elitescloud.cloudt.system.service.vo.SysUserPagingVO;
import com.elitescloud.cloudt.system.service.vo.component.SysUserComponentPagingVO;
import com.elitescloud.cloudt.system.spi.SysUserQuerySpi;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/4/8
 */
public class SysUserQueryProvider implements SysUserQuerySpi {

    private static final QSysUserDO Q_SYS_USER_DO = QSysUserDO.sysUserDO;
    private static final QSysTenantUserDO Q_SYS_TENANT_USER_DO = QSysTenantUserDO.sysTenantUserDO;

    @Autowired
    private JPAQueryFactory jpaQueryFactory;
    @Autowired
    private TenantClientProvider tenantClientProvider;

    @Override
    public String getServiceName() {
        return "租户服务";
    }

    @Override
    public JPAQuery<SysUserDO> selectSysUserForJpaQuery() {
        if (!tenantClientProvider.enabledTenant()) {
            // 未启用多租户
            return null;
        }

        GeneralUserDetails user = SecurityContextUtil.currentUser();
        if (!needFilter(user)) {
            return null;
        }

        return jpaQueryFactory.select(Q_SYS_USER_DO)
                .from(Q_SYS_USER_DO).leftJoin(Q_SYS_TENANT_USER_DO).on(Q_SYS_USER_DO.id.eq(Q_SYS_TENANT_USER_DO.sysUserId));
    }

    @Override
    public Predicate selectSysUserForPredicate() {
        if (!tenantClientProvider.enabledTenant()) {
            // 未启用多租户
            return null;
        }

        return filterByTenant();
    }

    @Override
    public JPAQuery<SysUserPagingVO> findPagingForJpaQuery() {
        if (!tenantClientProvider.enabledTenant()) {
            // 未启用多租户
            return null;
        }

        GeneralUserDetails user = SecurityContextUtil.currentUser();
        if (!needFilter(user)) {
            return null;
        }

        return jpaQueryFactory
                .select(Projections.bean(
                        SysUserPagingVO.class,
                        Q_SYS_USER_DO.id,
                        Q_SYS_USER_DO.username,
                        Q_SYS_USER_DO.lastName,
                        Q_SYS_USER_DO.firstName,
                        Q_SYS_USER_DO.mobile,
                        Q_SYS_USER_DO.email,
                        Q_SYS_USER_DO.enabled
                )).from(Q_SYS_USER_DO).leftJoin(Q_SYS_TENANT_USER_DO).on(Q_SYS_USER_DO.id.eq(Q_SYS_TENANT_USER_DO.sysUserId));
    }

    @Override
    public Predicate findPagingForPredicate() {
        if (!tenantClientProvider.enabledTenant()) {
            // 未启用多租户
            return null;
        }

        return filterByTenant();
    }

    @Override
    public JPAQuery<SysUserComponentPagingVO> findSysUserComPagingByParamForJpaQuery() {
        if (!tenantClientProvider.enabledTenant()) {
            // 未启用多租户
            return null;
        }

        GeneralUserDetails user = SecurityContextUtil.currentUser();
        if (!needFilter(user)) {
            return null;
        }

        return jpaQueryFactory
                .select(Projections.bean(
                        SysUserComponentPagingVO.class,
                        Q_SYS_USER_DO.id,
                        Q_SYS_USER_DO.username,
                        Q_SYS_USER_DO.lastName,
                        Q_SYS_USER_DO.firstName,
                        Q_SYS_USER_DO.mobile,
                        Q_SYS_USER_DO.email
                ))
                .from(Q_SYS_USER_DO).leftJoin(Q_SYS_TENANT_USER_DO).on(Q_SYS_USER_DO.id.eq(Q_SYS_TENANT_USER_DO.sysUserId));
    }

    @Override
    public Predicate findSysUserComPagingByParamForPredicate() {
        if (!tenantClientProvider.enabledTenant()) {
            // 未启用多租户
            return null;
        }

        return filterByTenant();
    }

    private boolean needFilter(GeneralUserDetails user) {
        if (user == null) {
            return false;
        }
        if (user.isSystemAdmin() || user.isOperation()) {
            // 管理员则直接用原来的
            return false;
        }

        return true;
    }

    private Predicate filterByTenant() {
        GeneralUserDetails user = SecurityContextUtil.currentUser();
        if (!needFilter(user)) {
            return null;
        }

        var tenant = user.getTenant();
        if (tenant == null) {
            throw new BusinessException("当前用户暂未绑定租户");
        }

        return Q_SYS_TENANT_USER_DO.sysTenantId.eq(tenant.getId()).and(Q_SYS_TENANT_USER_DO.sysUserId.ne(tenant.getSysUserId()));
    }
}
