package com.elitescloud.cloudt.tenant.service.impl;

import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.param.IdCodeNameCheckParam;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.core.app.PlatformAppProvider;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantAppDO;
import com.elitescloud.cloudt.system.service.repo.SysTenantAppRepo;
import com.elitescloud.cloudt.system.service.repo.SysTenantAppRepoProc;
import com.elitescloud.cloudt.tenant.service.SysTenantAppService;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/9/20
 */
@Log4j2
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class SysTenantAppServiceImpl implements SysTenantAppService {

    @Autowired
    private SysTenantAppRepo tenantAppRepo;
    @Autowired
    private SysTenantAppRepoProc tenantAppRepoProc;
    @Autowired
    private PlatformAppProvider platformAppProvider;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveApps(Long tenantId, Set<String> appCode) {
        if (tenantId == null) {
            return ApiResult.fail("租户ID为空");
        }

        // 目前已有的app
        Set<String> existsCodes = tenantAppRepoProc.getAppCode(tenantId);

        // 需要新增的app
        LocalDateTime now = LocalDateTime.now();
        List<SysTenantAppDO> codesToAdd = new ArrayList<>(appCode.size());
        SysTenantAppDO appDO = null;
        for (String s : appCode) {
            if (existsCodes.contains(s)) {
                continue;
            }
            appDO = new SysTenantAppDO();
            appDO.setAppCode(s);
            appDO.setSysTenantId(tenantId);
            appDO.setAssignTime(now);
            codesToAdd.add(appDO);
        }

        // 需要删除的
        Set<String> codesToDel = new HashSet<>(existsCodes.size());
        for (String s : existsCodes) {
            if (appCode.contains(s)) {
                continue;
            }
            codesToDel.add(s);
        }

        // 更新数据库
        if (!codesToAdd.isEmpty()) {
            tenantAppRepo.saveAll(codesToAdd);
        }
        if (!codesToDel.isEmpty()) {
            tenantAppRepoProc.deleteByTenantAndAppCode(tenantId, codesToDel);
        }

        return ApiResult.ok(tenantId);
    }

    @Override
    public ApiResult<List<IdCodeNameCheckParam>> getApps(Long tenantId) {
        if (tenantId == null) {
            return ApiResult.fail("租户ID为空");
        }

        Set<String> existsCodes = tenantAppRepoProc.getAppCode(tenantId);
        if (existsCodes.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        var apps = platformAppProvider.all();
        if (apps.isEmpty()) {
            return ApiResult.fail("获取平台应用失败");
        }
        var result = apps.stream()
                .map(t -> {
                    IdCodeNameCheckParam param = new IdCodeNameCheckParam();
                    param.setCode(t.getCode());
                    param.setName(t.getName());
                    param.setChecked(existsCodes.contains(t.getCode()));
                    return param;
                }).collect(Collectors.toList());

        return ApiResult.ok(result);
    }
}
