package com.elitescloud.cloudt.tenant.service.repo;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.common.BaseRepoProc;
import com.elitescloud.cloudt.system.service.model.entity.QSysTenantDO;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantDO;
import com.elitescloud.cloudt.tenant.model.vo.ApiTenantVO;
import com.elitescloud.cloudt.tenant.model.vo.SysTenantVO;
import com.elitescloud.cloudt.tenant.model.vo.params.ApiTenantQueryParamVO;
import com.elitescloud.cloudt.tenant.model.vo.params.SysTenantQueryParam;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import org.springframework.stereotype.Repository;

import java.util.List;

/**
 * <p>
 * 功能说明
 * </p>
 *
 * @author roman.zhang
 * @since 2022-02-16 15:16:02
 */
@Repository
public class SysTenantRepoProc extends BaseRepoProc<SysTenantDO> {

    private static final QSysTenantDO QDO = QSysTenantDO.sysTenantDO;

    public SysTenantRepoProc() {
        super(QDO);
    }

    public void bindDatasourceId(Long id, Long datasourceId) {
        jpaQueryFactory.update(QDO)
                .set(QDO.databaseSourceId, datasourceId)
                .where(QDO.id.eq(id))
                .execute();
    }

    public Long getDatasourceId(Long id) {
        return getValue(QDO.databaseSourceId, id);
    }

    /**
     * 启用、禁用租户
     *
     * @param id      租户ID
     * @param enabled 是否启用
     * @return
     */
    public boolean updateEnabled(Long id, boolean enabled) {
        return jpaQueryFactory.update(QDO)
                .set(QDO.enabled, enabled)
                .where(QDO.id.eq(id))
                .execute() > 0
                ;
    }

    /**
     * 更新数据库初始化结果
     *
     * @param id            租户ID
     * @param dbInitialized 是佛已初始化
     * @return
     */
    public boolean updateDbInitialized(Long id, boolean dbInitialized) {
        return jpaQueryFactory.update(QDO)
                .set(QDO.dbInitialized, dbInitialized)
                .where(QDO.id.eq(id))
                .execute() > 0
                ;
    }

    /**
     * 更新基础数据同步记录
     *
     * @param id
     * @param syncId
     */
    public void updateBaseDataSyncId(Long id, Long syncId) {
        super.updateValue(QDO.baseDataSyncId, syncId, id);
    }

    /**
     * 判断是否已初始化
     *
     * @param id
     * @return
     */
    public Boolean isDbInitialized(Long id) {
        return jpaQueryFactory.select(QDO.dbInitialized)
                .from(QDO)
                .where(QDO.id.eq(id))
                .fetchOne();
    }

    /**
     * 判断编号是否存在
     *
     * @param tenantCode 租户编号
     * @param excludeId  租户ID
     * @return 是否存在
     */
    public boolean existsTenantCode(String tenantCode, Long excludeId) {
        return exists(QDO.tenantCode, tenantCode, excludeId);
    }

    /**
     * 判断schemaName是否存在
     *
     * @param schemaName schemaName
     * @param excludeId  租户ID
     * @return 是否存在
     */
    public boolean existsSchemaName(String schemaName, Long excludeId) {
        return exists(QDO.schemaName, schemaName, excludeId);
    }

    /**
     * 判断域名是否存在
     *
     * @param tenantDomain 域名
     * @param excludeId    租户ID
     * @return 是否存在
     */
    public boolean existsTenantDomain(String tenantDomain, Long excludeId) {
        return exists(QDO.tenantDomain, tenantDomain, excludeId);
    }

    /**
     * 判断域名是否存在
     *
     * @param customDomain 域名
     * @param excludeId    租户ID
     * @return 是否存在
     */
    public boolean existsCustomDomain(String customDomain, Long excludeId) {
        return exists(QDO.customDomain, customDomain, excludeId);
    }

    /**
     * 查询租户信息
     *
     * @param param 查询参数
     * @return 租户信息
     */
    public PagingVO<SysTenantVO> query(SysTenantQueryParam param) {
        var predicate = PredicateBuilder.builder()
                .andOnNotBlank(param.getKeyword(), () -> QDO.tenantCode.like("%" + param.getKeyword() + "%").or(QDO.tenantName.like("%" + param.getKeyword() + "%")))
                .andOnNotBlank(param.getTenantCode(), () -> QDO.tenantCode.like("%" + param.getTenantCode() + "%"))
                .andOnNotBlank(param.getTenantName(), () -> QDO.tenantName.like("%" + param.getTenantName() + "%"))
                .andOnNotNull(param.getEnabled(), () -> QDO.enabled.eq(param.getEnabled()))
                .andOnNotBlank(param.getLinkman(), () -> QDO.linkman.like("%" + param.getLinkman() + "%"))
                .andOnNotBlank(param.getContactNumber(), () -> QDO.contactNumber.like("%" + param.getContactNumber() + "%"))
                .andOnNotBlank(param.getSchemaName(), () -> QDO.schemaName.like("%" + param.getSchemaName() + "%"))
                .andOnNotBlank(param.getTenantDomain(), () -> QDO.tenantDomain.like("%" + param.getTenantDomain() + "%"))
                .andOnNotBlank(param.getCustomDomain(), () -> QDO.customDomain.like("%" + param.getCustomDomain() + "%"))
                .andOnNotBlank(param.getIndustry(), () -> QDO.industry.eq(param.getIndustry()))
                .andOnNotBlank(param.getCustomer(), () -> QDO.customer.eq(param.getCustomer()))
                .build();

        var jpaQuery = jpaQueryFactory.select(fieldsOfSysTenantVO())
                .from(QDO)
                .where(predicate);
        return queryByPage(jpaQuery, param.getPageRequest());
    }

    /**
     * 查询所有租户
     *
     * @return
     */
    public List<ApiTenantVO> allApi() {
        return jpaQueryFactory.select(fieldsOfApiTenantVO())
                .from(QDO)
                .where(QDO.enabled.eq(true))
                .fetch();
    }

    /**
     * 分页查询租户
     *
     * @param paramVO
     * @return
     */
    public PagingVO<ApiTenantVO> queryApi(ApiTenantQueryParamVO paramVO) {
        Predicate predicate = PredicateBuilder.builder()
                .andOnNotBlank(paramVO.getTenantCode(), () -> QDO.tenantCode.eq(paramVO.getTenantCode()))
                .andOnNotBlank(paramVO.getTenantName(), () -> QDO.tenantName.like("%" + paramVO.getTenantName() + "%"))
                .andOnNotBlank(paramVO.getKeyword(), () -> QDO.tenantName.like("%" + paramVO.getKeyword() + "%"))
                .build();
        var jpaQuery = jpaQueryFactory.select(fieldsOfApiTenantVO())
                .from(QDO)
                .where(predicate);
        return queryByPage(jpaQuery, paramVO.getPageRequest());
    }

    private static QBean<SysTenantVO> fieldsOfSysTenantVO() {
        return Projections.bean(SysTenantVO.class, QDO.id, QDO.tenantCode, QDO.tenantName, QDO.enabled, QDO.adminAccount, QDO.sysUserId.as("adminUserId"), QDO.linkman,
                QDO.contactNumber, QDO.address, QDO.tenantIsolation, QDO.dbInitialized, QDO.databaseSourceId, QDO.schemaName, QDO.tenantDomain,
                QDO.customDomain, QDO.industry, QDO.customer, QDO.remark, QDO.createTime, QDO.creator, QDO.modifyTime, QDO.updater);
    }

    private static QBean<ApiTenantVO> fieldsOfApiTenantVO() {
        return Projections.bean(ApiTenantVO.class, QDO.id, QDO.tenantCode, QDO.tenantName, QDO.linkman, QDO.contactNumber,
                QDO.address, QDO.tenantDomain, QDO.customDomain, QDO.industry, QDO.customer);
    }
}

