package com.elitescloud.cloudt.tenant.service.impl;

import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.constant.TenantIsolateStrategy;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantDO;
import com.elitescloud.cloudt.tenant.config.TenantProperties;
import com.elitescloud.cloudt.tenant.convert.SysTenantDbMigrateConvert;
import com.elitescloud.cloudt.tenant.model.entity.SysTenantDbMigrateDO;
import com.elitescloud.cloudt.tenant.model.vo.SysTenantDbMigrateVO;
import com.elitescloud.cloudt.tenant.service.SysTenantDbMigrateService;
import com.elitescloud.cloudt.tenant.service.manager.SysTenantManager;
import com.elitescloud.cloudt.tenant.service.repo.SysTenantDbMigrateRepo;
import com.elitescloud.cloudt.tenant.service.repo.SysTenantDbMigrateRepoProc;
import com.elitescloud.cloudt.tenant.service.repo.SysTenantRepoProc;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.*;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/3/28
 */
@Log4j2
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class SysTenantDbMigrateServiceImpl implements SysTenantDbMigrateService {

    private static final SysTenantDbMigrateConvert CONVERT = SysTenantDbMigrateConvert.INSTANCE;

    @Autowired
    private TenantProperties tenantProperties;
    @Autowired
    private SysTenantDbMigrateRepo tenantDbMigrateRepo;
    @Autowired
    private SysTenantDbMigrateRepoProc tenantDbMigrateRepoProc;
    @Autowired
    private SysTenantRepoProc tenantRepoProc;
    @Autowired
    private SysTenantManager tenantManager;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> addInit(SysTenantDO tenantDO, TenantIsolateStrategy isolateStrategy, String appCode) {
        // 先判断是否已存在
        SysTenantDbMigrateDO migrateDO = tenantDbMigrateRepoProc.getByTenantAndAppCode(tenantDO.getId(), appCode);
        if (migrateDO != null) {
            return ApiResult.ok(migrateDO.getId());
        }

        migrateDO = new SysTenantDbMigrateDO();
        migrateDO.setSysTenantId(tenantDO.getId());
        migrateDO.setAppCode(appCode);
        migrateDO.setTenantIsolation(isolateStrategy.name());
        migrateDO.setDatabaseSourceId(tenantDO.getDatabaseSourceId());
        migrateDO.setSchemaName(tenantDO.getSchemaName());
        migrateDO.setMigrateSuccess(false);
        migrateDO.setMigrateStartTime(LocalDateTime.now());
        tenantDbMigrateRepo.save(migrateDO);
        return ApiResult.ok(migrateDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Set<String>> updateMigrateResultForTenant(Long tenantId, TenantIsolateStrategy isolateStrategy, Set<String> updatedApps) {
        if (tenantProperties.isDiscoveryClient() || tenantProperties.getClientNames().isEmpty()) {
            // 都已更新
            tenantRepoProc.updateDbInitialized(tenantId, true);

            return ApiResult.ok(Collections.emptySet());
        }

        LocalDateTime now = LocalDateTime.now();
        Set<String> failApps = new HashSet<>();
        List<SysTenantDbMigrateDO> migrateDos = new ArrayList<>();
        for (String clientName : tenantProperties.getClientNames()) {
            if (updatedApps.contains(clientName)) {
                continue;
            }

            SysTenantDbMigrateDO migrateDO = new SysTenantDbMigrateDO();
            migrateDO.setSysTenantId(tenantId);
            migrateDO.setAppCode(clientName);
            migrateDO.setTenantIsolation(isolateStrategy.name());
            migrateDO.setMigrateSuccess(false);
            migrateDO.setMigrateStartTime(now);
            migrateDO.setMigrateFinishTime(now);
            migrateDO.setFailReason("服务不在线");

            migrateDos.add(migrateDO);
            failApps.add(clientName);
        }

        if (migrateDos.isEmpty()) {
            // 都已更新
            tenantRepoProc.updateDbInitialized(tenantId, true);
        } else {
            tenantDbMigrateRepo.saveAll(migrateDos);
        }

        return ApiResult.ok(failApps);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateResult(Long id, boolean success, String failReason) {
        tenantDbMigrateRepoProc.updateResult(id, success, failReason);

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> createTenantDb(Long tenantId) {
        var tenant = tenantRepoProc.get(tenantId);
        if (tenant == null) {
            return ApiResult.fail("租户不存在");
        }
        if (Boolean.TRUE.equals(tenant.getDbInitialized())) {
            return ApiResult.fail("数据库已初始化");
        }

        tenantManager.syncClientDb(tenant);

        return ApiResult.ok(tenantId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> retry(Long id) {
        SysTenantDbMigrateDO migrateDO = tenantDbMigrateRepoProc.get(id);
        if (migrateDO == null) {
            return ApiResult.fail("记录不存在");
        }

        boolean retryResult = tenantManager.retryDbMigrate(migrateDO);
        if (retryResult) {
            updateTenantDbInitialized(migrateDO.getSysTenantId());
        }
        return retryResult ? ApiResult.ok(id) : ApiResult.fail("操作失败");
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> deleteTenantDb(Long tenantId) {
        SysTenantDO tenant = tenantRepoProc.get(tenantId);
        if (tenant == null) {
            return ApiResult.fail("租户不存在");
        }

        tenantManager.deleteClientDb(tenant);
        return ApiResult.ok(tenantId);
    }

    @Override
    public ApiResult<List<SysTenantDbMigrateVO>> queryByTenant(Long tenantId) {
        var migrateDos = tenantDbMigrateRepoProc.queryByTenantId(tenantId);
        if (migrateDos.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        var vos = CONVERT.dos2Vos(migrateDos);
        return ApiResult.ok(vos);
    }

    private void updateTenantDbInitialized(Long tenantId) {
        long failNum = tenantDbMigrateRepoProc.countFailNum(tenantId);
        if (failNum == 0) {
            // 全部已更新
            tenantRepoProc.updateDbInitialized(tenantId, true);
        }
    }

    @Override
    public ApiResult<Long> getMigrateId(Long tenantId, String appCode) {
        SysTenantDbMigrateDO migrateDO = tenantDbMigrateRepoProc.getByTenantAndAppCode(tenantId, appCode);
        if (migrateDO != null) {
            return ApiResult.ok(migrateDO.getId());
        }
        return ApiResult.ok(null);
    }
}
