package com.elitescloud.cloudt.tenant.service.impl;

import com.elitescloud.boot.common.param.IdCodeNameCheckParam;
import com.elitescloud.boot.provider.PlatformAppProvider;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.system.constant.TenantType;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantDO;
import com.elitescloud.cloudt.system.service.repo.SysTenantAppRepoProc;
import com.elitescloud.cloudt.system.service.repo.SysTenantUserRepoProc;
import com.elitescloud.cloudt.tenant.convert.SysTenantConvert;
import com.elitescloud.cloudt.tenant.service.SysTenantQueryService;
import com.elitescloud.cloudt.tenant.service.repo.SysTenantRepoProc;
import lombok.RequiredArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/28
 */
@Log4j2
@RequiredArgsConstructor
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class SysTenantQueryServiceImpl extends BaseServiceImpl implements SysTenantQueryService {
    private static final SysTenantConvert CONVERT = SysTenantConvert.INSTANCE;

    @Autowired
    private SysTenantRepoProc sysTenantRepoProc;
    @Autowired
    private SysTenantAppRepoProc tenantAppRepoProc;
    @Autowired
    private SysTenantUserRepoProc sysTenantUserRepoProc;

    @Autowired
    private PlatformAppProvider platformAppProvider;

    @Override
    public ApiResult<List<SysTenantDTO>> allTenants() {
        var tenantDos = sysTenantRepoProc.allForUnDeleted();
        if (tenantDos.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 查询租户的应用
        var appMap = tenantAppRepoProc.getAppCodeAll();

        var tenants = tenantDos.stream()
                .filter(t -> Boolean.TRUE.equals(t.getEnabled()))
                .map(t -> {
                    var dto = this.do2Dto(t);
                    dto.setAppCodes(appMap.get(t.getId()));
                    return dto;
                })
                .collect(Collectors.toList());

        return ApiResult.ok(tenants);
    }

    @Override
    public ApiResult<SysTenantDTO> getDtoById(Long id) {
        var tenantDO = sysTenantRepoProc.get(id);
        if (tenantDO == null) {
            return ApiResult.fail("租户不存在");
        }
        if (!Boolean.TRUE.equals(tenantDO.getEnabled())) {
            return ApiResult.fail("租户已禁用");
        }

        var dto = this.do2Dto(tenantDO);
        dto.setAppCodes(tenantAppRepoProc.getAppCode(id));
        return ApiResult.ok(dto);
    }

    @Override
    public ApiResult<List<SysTenantDTO>> queryTenantDtoOfUser(Long sysUserId) {
        var tenantDoList = sysTenantUserRepoProc.queryUserTenantDetail(sysUserId);
        if (tenantDoList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 查询租户的应用
        Set<Long> tenantIds = tenantDoList.stream().map(SysTenantDO::getId).collect(Collectors.toSet());
        var tenantApps = tenantAppRepoProc.getAppCodes(tenantIds);

        var tenants = tenantDoList.stream()
                .filter(t -> Boolean.TRUE.equals(t.getEnabled()))
                .map(t -> {
                    var dto = this.do2Dto(t);
                    dto.setAppCodes(tenantApps.getOrDefault(t.getId(), Collections.emptySet()));
                    return dto;
                })
                .collect(Collectors.toList());
        return ApiResult.ok(tenants);
    }

    @Override
    public ApiResult<List<IdCodeNameCheckParam>> getApps(Long tenantId) {
        if (tenantId == null) {
            return ApiResult.fail("租户ID为空");
        }

        // 获取所有应用列表
        var apps = platformAppProvider.all();
        if (apps.isEmpty()) {
            return ApiResult.fail("未获取到有效的应用列表");
        }
        // 获取租户的应用
        Set<String> existsCodes = tenantAppRepoProc.getAppCode(tenantId);
        // 返回结果
        var result = apps.stream()
                .map(t -> {
                    IdCodeNameCheckParam param = new IdCodeNameCheckParam();
                    param.setCode(t.getCode());
                    param.setName(t.getName());
                    param.setChecked(existsCodes.contains(t.getCode()));
                    return param;
                }).collect(Collectors.toList());

        return ApiResult.ok(result);
    }

    private SysTenantDTO do2Dto(SysTenantDO tenantDO) {
        var dto = CONVERT.do2Dto(tenantDO);
        dto.setType(TenantType.parse(tenantDO.getTenantType()));
        dto.setIsolationStrategy(SysTenantMngServiceImpl.convertTenantIsolate(tenantDO.getTenantIsolation()));
        return dto;
    }
}
