package com.elitescloud.cloudt.tenant.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.tenant.client.common.TenantClient;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.service.callback.UserChangedCallback;
import com.elitescloud.cloudt.system.service.model.bo.SysUserSaveBO;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantUserDO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserDO;
import com.elitescloud.cloudt.system.service.repo.SysTenantUserRepo;
import com.elitescloud.cloudt.system.service.repo.SysTenantUserRepoProc;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;

/**
 * 租户用户相关回调服务.
 *
 * @author Kaiser（wang shao）
 * 2022/9/28
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
@Log4j2
public class UserChangedCallbackTenantServiceImpl implements UserChangedCallback {

    @Autowired
    private TenantClientProvider tenantClientProvider;
    @Autowired
    private SysTenantUserRepo tenantUserRepo;
    @Autowired
    private SysTenantUserRepoProc tenantUserRepoProc;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void onUpsert(boolean add, SysUserSaveBO saveBO, SysUserDO userDO) {
        if (!tenantClientProvider.enabledTenant()) {
            log.info("未启用租户功能");
            return;
        }
        // 保存用户与租户关系
        if (add) {
            onInsert(saveBO, userDO);
            return;
        }
        onUpdate(saveBO, userDO);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void onEnabled(Long userId, boolean enabled) {
        if (!tenantClientProvider.enabledTenant()) {
            log.info("未启用租户功能");
            return;
        }

        Long tenantId = currentTenantId();
        tenantUserRepoProc.updateEnabled(tenantId, userId, enabled);
    }

    @Override
    public void onUpdatePassword(Long userId, String password) {

    }

    @Override
    public void onUpdateMobile(Long userId, String mobile) {

    }

    @Override
    public void onUpdateEmail(Long userId, String email) {

    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void onDelete(Long userId) {
        if (!tenantClientProvider.enabledTenant()) {
            log.info("未启用租户功能");
            return;
        }

        Long tenantId = currentTenantId();
        tenantUserRepoProc.unbindTenant(tenantId, userId);
    }

    private void onInsert(SysUserSaveBO saveBO, SysUserDO userDO) {
        Long tenantId = currentTenantId();
        log.info("绑定用户{}与租户{}", userDO.getId(), tenantId);

        SysTenantUserDO tenantUserDO = new SysTenantUserDO();
        tenantUserDO.setSysUserId(userDO.getId());
        tenantUserDO.setSysTenantId(tenantId);
        tenantUserDO.setBindTime(LocalDateTime.now());
        tenantUserDO.setExpiredTime(saveBO.getExpiredTime());
        tenantUserDO.setEnabled(ObjectUtil.defaultIfNull(saveBO.getEnabled(), true));
        tenantUserDO.setDef(false);

        tenantUserRepo.save(tenantUserDO);
    }

    private void onUpdate(SysUserSaveBO saveBO, SysUserDO userDO) {
        Long tenantId = currentTenantId();

        SysTenantUserDO tenantUserDO = tenantUserRepoProc.get(tenantId, userDO.getId());
        if (tenantUserDO == null) {
            // 不存在，则新绑定
            onInsert(saveBO, userDO);
            return;
        }
        tenantUserDO.setExpiredTime(saveBO.getExpiredTime());
        tenantUserDO.setEnabled(saveBO.getEnabled());

        tenantUserRepo.save(tenantUserDO);
    }

    private Long currentTenantId() {
        var currentTenant = TenantClient.getCurrentTenant();
        return currentTenant == null ? TenantConstant.DEFAULT_TENANT_ID : currentTenant.getId();
    }
}
