package com.elitescloud.cloudt.tenant.controller;

import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.tenant.model.vo.params.SysTenantDatasourceRespVO;
import com.elitescloud.cloudt.tenant.model.vo.params.SysTenantDatasourceSaveVO;
import com.elitescloud.cloudt.tenant.service.SysTenantDatasourceService;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * 租户数据源管理.
 *
 * @author Kaiser（wang shao）
 * @date 2023/9/27
 */
@ResponseBody
@RequestMapping("/sys/tenantDs")
@Api(tags = {"租户数据源管理"})
@Validated
public class SysTenantDatasourceController {

    @Autowired
    private SysTenantDatasourceService service;

    /**
     * 可分配数据源的应用列表
     *
     * @return 应用列表
     */
    @ApiOperation(value = "可分配数据源的应用列表")
    @ApiOperationSupport(order = 1)
    @ApiImplicitParam(name = "sysTenantId", value = "租户ID")
    @GetMapping(value = "/app")
    public ApiResult<List<CodeNameParam>> appList(@RequestParam(name = "sysTenantId") Long sysTenantId) {
        return service.appList(sysTenantId);
    }

    /**
     * 保存租户数据源
     *
     * @param saveVO 数据源信息
     * @return 数据源ID
     */
    @ApiOperation(value = "保存租户数据源")
    @ApiOperationSupport(order = 2)
    @PostMapping(value = "/save")
    public ApiResult<Long> save(@RequestBody @Valid SysTenantDatasourceSaveVO saveVO) {
        return service.save(saveVO);
    }

    /**
     * 测试连接
     *
     * @param saveVO 数据源信息
     * @return 数据源ID
     */
    @ApiOperation(value = "测试连接")
    @ApiOperationSupport(order = 3)
    @PostMapping(value = "/test/connection")
    public ApiResult<String> testConnection(@RequestBody @Valid SysTenantDatasourceSaveVO saveVO) {
        return service.testConnection(saveVO);
    }

    /**
     * 获取数据源配置
     *
     * @param sysTenantId 租户ID
     * @param appCode     应用编码
     * @return 数据源配置
     */
    @ApiOperation(value = "获取数据源配置")
    @ApiOperationSupport(order = 4)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "sysTenantId", value = "租户ID", required = true),
            @ApiImplicitParam(name = "appCode", value = "应用编码", required = true),
    })
    @GetMapping(value = "/get")
    public ApiResult<SysTenantDatasourceRespVO> get(@NotNull(message = "租户ID为空") @RequestParam(name = "sysTenantId") Long sysTenantId,
                                                    @NotBlank(message = "应用编码为空") @RequestParam(name = "appCode") String appCode) {
        return service.get(sysTenantId, appCode);
    }

}
