package com.elitesland.cbpl.infinity.client.account.interceptor;

import cn.hutool.extra.spring.SpringUtil;
import com.elitescloud.cloudt.common.common.CloudtOptional;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitescloud.cloudt.tenant.config.support.DefaultTenantProvider;
import com.elitescloud.cloudt.tenant.config.support.TenantSession;
import com.elitesland.cbpl.infinity.client.track.service.InfinityTrackService;
import com.elitesland.cbpl.infinity.client.track.service.InfinityTrackServiceImpl;
import com.elitesland.cbpl.infinity.server.account.vo.resp.InfinityAccountRespVO;
import com.elitesland.cbpl.util.ExceptionUtil;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.protocol.HTTP;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Optional;

/**
 * @author eric.hao
 * @since 2023/05/30
 */
@Slf4j
@Data
@NoArgsConstructor
public class AuthInterceptor implements HandlerInterceptor {

    private List<InfinityAccountRespVO> accounts;

    public AuthInterceptor(List<InfinityAccountRespVO> accounts) {
        this.accounts = accounts;
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) {
        response.addHeader(HTTP.CONTENT_TYPE, "application/json;charset=UTF-8");
        InfinityTrackService trackService = SpringUtil.getBean(InfinityTrackServiceImpl.class);
        try {
            // 记录调用日志
            // get header token
            final String requestMethod = request.getMethod();
            final String token = request.getHeader("Authorization");
            log.info("checkApiToken: check token,request method={},header token={}", requestMethod, token);
            // token
            if (StringUtils.isBlank(token)) {
                response.setStatus(401);
                log.error("请求参数非法，请核实!");
                trackService.error(request, "【401：请求参数非法】");
                return false;
            }
            Optional<InfinityAccountRespVO> matchAccount = accounts.stream()
                    .filter(account -> checkSignature(token, account.getUsername(), account.getPassword())).findFirst();
            if (matchAccount.isEmpty()) {
                response.setStatus(203);
                log.error("checkApiToken: authorization invalid,head Authorization={}", token);
                trackService.error(request, "【203：认证失败】");
                return false;
            }
            // 设置当前租户
            SysTenantDTO tenant = setCurrentTenant(matchAccount.get());
            trackService.log(request, tenant);
            return true;
        } catch (Exception e) {
            log.error("checkApiToken: catch e.", e);
            response.setStatus(500);
            trackService.error(request, ExceptionUtil.formatException(e));
            return false;
        } finally {
        }
    }

    @Override
    public void postHandle(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse, Object o, ModelAndView modelAndView) throws Exception {
        resetCurrentTenant();
    }

    @Override
    public void afterCompletion(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse, Object o, Exception e) throws Exception {

    }

    private boolean checkSignature(String authorization, String username, String password) {
        String auth = username + ":" + password;
        byte[] encodedAuth = Base64.encodeBase64((auth.getBytes(StandardCharsets.US_ASCII)));
        String authHeader = "Basic " + new String(encodedAuth);
        return authHeader.equals(authorization);
    }

    private SysTenantDTO hasUsedTenant;
    private boolean hasUsedDefault;
    private boolean hasNot;

    private SysTenantDTO setCurrentTenant(InfinityAccountRespVO account) {
        // 如果之前不使用租户，则先取消
        var hasNot = TenantSession.getNoTenant();
        if (hasNot) {
            TenantSession.clearNoTenant();
        }

        // 如果之前使用默认的，则先取消使用默认
        hasUsedDefault = TenantSession.getUseDefault();
        if (hasUsedDefault) {
            TenantSession.clearUseDefault();
        }

        // 如果之前设置了session的租户，则先更换
        hasUsedTenant = TenantSession.getCurrentTenant();

        // 设置了对应租户，设置切换
        DefaultTenantProvider tenantProvider = SpringUtil.getBean(DefaultTenantProvider.class);
        CloudtOptional<SysTenantDTO> tenantDTO = tenantProvider.getById(account.getBelongTenant());
        TenantSession.setCurrentTenant(tenantDTO.get());
        return tenantDTO.get();
    }

    private void resetCurrentTenant() {
        // 恢复原session租户
        if (hasUsedTenant == null) {
            TenantSession.clearCurrentTenant();
        } else {
            TenantSession.setCurrentTenant(hasUsedTenant);
        }

        // 恢复原是否使用默认
        if (hasUsedDefault) {
            TenantSession.setUseDefault();
        }

        // 恢复不使用
        if (hasNot) {
            TenantSession.setNoTenant();
        }
    }
}
