package com.elitesland.cbpl.infinity.client.track.service;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.extra.servlet.ServletUtil;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.tool.es.repository.ElasticsearchRepository;
import com.elitesland.cbpl.tool.es.service.ElasticTrackService;
import com.elitesland.cbpl.tool.es.util.ElasticsearchUtil;
import com.elitesland.cbpl.util.ExceptionUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import static com.elitesland.cbpl.infinity.client.config.InfinityClientConfig.INFINITY_POLICY_NAME;
import static com.elitesland.cbpl.infinity.client.config.InfinityClientConfig.INFINITY_ROLLOVER_ALIAS;
import static com.elitesland.cbpl.tool.es.domain.TrackEvent.TRACK_INFINITY;

/**
 * @author eric.hao
 * @since 2023/06/13
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class InfinityTrackServiceImpl implements InfinityTrackService {

    private static final String IDX_INFINITY = "infinity_track";

    @Resource
    private TaskExecutor taskExecutor;
    @Resource
    private ElasticsearchRepository elasticRepository;
    private final ElasticTrackService elasticTrackService;

    @Override
    public void log(HttpServletRequest request, SysTenantDTO tenant) {
        addLog(request, tenant, "OK");
    }

    @Override
    public void error(HttpServletRequest request, String errorMessage) {
        addLog(request, null, errorMessage);
    }

    private void addLog(HttpServletRequest request, SysTenantDTO tenant, String errorMessage) {
        CompletableFuture.runAsync(() -> {
            Map<String, Object> param = new HashMap<>();
            param.put("CURRENT_TENANT", ObjectUtil.isNull(tenant) ? "-1" : tenant.getId().toString());
            param.put("REQUEST_URL", request.getRequestURL());
            param.put("REQUEST_METHOD", request.getMethod());
            param.put("REQUEST_HEADER", HttpServletUtil.getHeaders(request));
            param.put("REQUEST_PARAMS", ServletUtil.getParams(request));
            String bodyJsonStr = BeanUtils.toJsonStr(BeanUtils.toBean(ServletUtil.getBody(request), Map.class));
            param.put("REQUEST_BODY", bodyJsonStr);
            param.put("ERROR_MESSAGE", errorMessage);
            String indexName = ElasticsearchUtil.indexNameByDay(IDX_INFINITY);
            checkIndex(indexName);
            elasticTrackService.addTrackLog(request, indexName, TRACK_INFINITY.name(), param);
        }, taskExecutor).whenComplete((resp, ex) -> {
            if (ObjectUtil.isNotNull(ex)) {
                log.error(ExceptionUtil.formatException(ex));
            }
        });
    }

    /**
     * 检查索引是否存在
     * 如果不存在则创建，同时分配索引策略，默认
     *
     * @param indexName 索引名称
     */
    private void checkIndex(String indexName) {
        boolean exists = elasticRepository.existsIndex(indexName);
        if (!exists) {
            String policy = StrUtil.blankToDefault(INFINITY_POLICY_NAME, "");
            String alias = StrUtil.blankToDefault(INFINITY_ROLLOVER_ALIAS, "");
            elasticRepository.createIndex(indexName, policy, alias);
        }
    }
}
