package com.elitesland.cbpl.infinity.client.router.service;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.http.HttpUtil;
import com.elitesland.cbpl.infinity.client.router.builder.RouterParamBuilder;
import com.elitesland.cbpl.infinity.server.router.vo.param.InfinityRouterParamVO;
import com.elitesland.cbpl.infinity.web.common.vo.ResponseVO;
import com.elitesland.cbpl.infinity.web.http.param.HttpParam;
import com.elitesland.cbpl.infinity.web.http.service.RestClient;
import com.elitesland.cbpl.infinity.web.util.BeanConvertUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * @author eric.hao
 * @since 2023/04/06
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class RouterClientServiceImpl implements RouterClientService {

    private final RouterParamBuilder routerParamBuilder;
    private final RestClient restClient;
    @Autowired
    private RestTemplate restTemplate;

    @Override
    public <T, K extends ResponseVO> List<T> query(InfinityRouterParamVO param, Map<String, Object> requestBody, Class<T> resultType, Class<K> responseType) {
        HttpParam<T, K> routerParam = routerParamBuilder.routerParamBuilder(param, requestBody, resultType, responseType);
        try {
            K response = restClient.exchangeToBean(routerParam);
            if (ResponseVO.success(response, "S")) {
                List<Map<String, Object>> entities = response.getData();
                log.trace("[HTTP-REQ] RESULT LIST: {}", entities);
                return BeanConvertUtil.toEntities(entities, routerParam.getResultType());
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        log.error("[HTTP-REQ] QUERY FAIL. URL({}), PARAMS:({})", routerParam.getUrl(), BeanConvertUtil.hashMapToString(routerParam.getBody()));
        return Collections.emptyList();
    }

    @Override
    public <K extends ResponseVO> K send(InfinityRouterParamVO param, Map<String, Object> requestBody, Class<K> responseType) {
        String message;
        HttpParam<Object, K> routerParam = routerParamBuilder.routerParamBuilder(param, requestBody, null, responseType);
        try {
            return restClient.exchangeToBean(routerParam);
        } catch (Exception e) {
            message = e.getMessage();
            e.printStackTrace();
        }
        log.error("[HTTP-REQ] PUT FAIL. URL({}), PARAMS:({})", routerParam.getUrl(), BeanConvertUtil.hashMapToString(routerParam.getBody()));
        return BeanUtil.toBean(ResponseVO.error(message), responseType);
    }

    @Override
    public String exchange(InfinityRouterParamVO param, Map<String, Object> requestBody) {
        var routerParam = routerParamBuilder.routerParamBuilder(param, requestBody, null, null);
        try {
            return restClient.exchange(routerParam);
        } catch (Exception e) {
            e.printStackTrace();
        }
        log.error("[HTTP-REQ] PUT FAIL. URL({}), PARAMS:({})", routerParam.getUrl(), BeanConvertUtil.hashMapToString(routerParam.getBody()));
        return "";
    }

    @Override
    public Resource exchangeStream(InfinityRouterParamVO param, Map<String, Object> requestBody) {
        var routerParam = routerParamBuilder.routerParamBuilder(param, requestBody, null, null);
        try {
            String queryParams = HttpUtil.toParams(routerParam.getBody(), StandardCharsets.UTF_8);
            String requestUrl = routerParam.getUrl() + "?" + queryParams;
            HttpEntity<String> entity = restClient.getMethodHttpEntity(routerParam.getHeaders());
            return restTemplate.exchange(URI.create(requestUrl), HttpMethod.GET, entity, Resource.class).getBody();
        } catch (Exception e) {
            e.printStackTrace();
        }
        log.error("[HTTP-REQ] PUT FAIL. URL({}), PARAMS:({})", routerParam.getUrl(), BeanConvertUtil.hashMapToString(routerParam.getBody()));
        return null;
    }
}
