package com.elitesland.cbpl.infinity.client.account.interceptor;

import cn.hutool.extra.spring.SpringUtil;
import com.elitescloud.cloudt.common.common.CloudtOptional;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitescloud.cloudt.tenant.config.support.DefaultTenantProvider;
import com.elitescloud.cloudt.tenant.config.support.TenantSession;
import com.elitesland.cbpl.infinity.client.util.InfinitySession;
import com.elitesland.cbpl.infinity.server.account.vo.resp.InfinityAccountRespVO;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.protocol.HTTP;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import javax.annotation.Nonnull;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.List;
import java.util.Optional;

import static com.elitesland.cbpl.infinity.client.router.util.InfinityUtil.checkSignature;
import static java.net.HttpURLConnection.*;
import static org.springframework.http.MediaType.APPLICATION_JSON_VALUE;

/**
 * @author eric.hao
 * @since 2023/05/30
 */
@Slf4j
@Data
@NoArgsConstructor
public class AuthInterceptor implements HandlerInterceptor {

    private List<InfinityAccountRespVO> accounts;

    public AuthInterceptor(List<InfinityAccountRespVO> accounts) {
        this.accounts = accounts;
    }

    @Override
    public boolean preHandle(@Nonnull HttpServletRequest request, @Nonnull HttpServletResponse response, @Nonnull Object handler) {
        response.addHeader(HTTP.CONTENT_TYPE, APPLICATION_JSON_VALUE);
        try {
            // get header token
            final String requestMethod = request.getMethod();
            final String token = request.getHeader("Authorization");
            log.info("[INFINITY] checkApiToken: check token,request method={}, header token={}", requestMethod, token);
            // token
            if (StringUtils.isBlank(token)) {
                response.setStatus(HTTP_UNAUTHORIZED);
                log.error("[INFINITY] 请求参数非法，请核实!");
                return false;
            }
            Optional<InfinityAccountRespVO> matchAccount = accounts.stream()
                    .filter(account -> checkSignature(token, account.getUsername(), account.getPassword())).findFirst();
            if (matchAccount.isEmpty()) {
                response.setStatus(HTTP_NOT_AUTHORITATIVE);
                log.error("[INFINITY] checkApiToken: authorization invalid, head Authorization={}", token);
                return false;
            }
            // 设置当前Infinity账户
            InfinitySession.setCurrentAccount(matchAccount.get());
            // 设置当前租户
            setCurrentTenant(matchAccount.get());
            return true;
        } catch (Exception e) {
            log.error("[INFINITY] checkApiToken: catch e.", e);
            response.setStatus(HTTP_INTERNAL_ERROR);
            return false;
        } finally {
        }
    }

    @Override
    public void postHandle(@Nonnull HttpServletRequest request, @Nonnull HttpServletResponse response, @Nonnull Object handler, ModelAndView modelAndView) {
        resetCurrentTenant();
        InfinitySession.clearCurrentAccount();
    }

    @Override
    public void afterCompletion(@Nonnull HttpServletRequest request, @Nonnull HttpServletResponse response, @Nonnull Object handler, Exception e) {
    }

    private SysTenantDTO hasUsedTenant;
    private boolean hasUsedDefault;
    private boolean hasNot;

    private SysTenantDTO setCurrentTenant(InfinityAccountRespVO account) {
        // 如果之前不使用租户，则先取消
        var hasNot = TenantSession.getNoTenant();
        if (hasNot) {
            TenantSession.clearNoTenant();
        }

        // 如果之前使用默认的，则先取消使用默认
        hasUsedDefault = TenantSession.getUseDefault();
        if (hasUsedDefault) {
            TenantSession.clearUseDefault();
        }

        // 如果之前设置了session的租户，则先更换
        hasUsedTenant = TenantSession.getCurrentTenant();

        // 设置了对应租户，设置切换
        DefaultTenantProvider tenantProvider = SpringUtil.getBean(DefaultTenantProvider.class);
        CloudtOptional<SysTenantDTO> tenantDTO = tenantProvider.getById(account.getBelongTenant());
        TenantSession.setCurrentTenant(tenantDTO.get());
        return tenantDTO.get();
    }

    private void resetCurrentTenant() {
        // 恢复原session租户
        if (hasUsedTenant == null) {
            TenantSession.clearCurrentTenant();
        } else {
            TenantSession.setCurrentTenant(hasUsedTenant);
        }

        // 恢复原是否使用默认
        if (hasUsedDefault) {
            TenantSession.setUseDefault();
        }

        // 恢复不使用
        if (hasNot) {
            TenantSession.setNoTenant();
        }
    }
}
