package com.elitescloud.cloudt.lowcode.dynamic.service.dynamic.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.PlatformAppProvider;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.Application;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.lowcode.dynamic.common.*;
import com.elitescloud.cloudt.lowcode.dynamic.convert.DynamicConfiguratorConvert;
import com.elitescloud.cloudt.lowcode.dynamic.model.bo.DynamicModelColumnBO;
import com.elitescloud.cloudt.lowcode.dynamic.model.bo.DynamicModelTableBO;
import com.elitescloud.cloudt.lowcode.dynamic.model.convert.BoModelFieldTypeEnum;
import com.elitescloud.cloudt.lowcode.dynamic.model.entity.*;
import com.elitescloud.cloudt.lowcode.dynamic.model.vo.query.DynamicConfiguratorPageQueryVO;
import com.elitescloud.cloudt.lowcode.dynamic.model.vo.resp.*;
import com.elitescloud.cloudt.lowcode.dynamic.model.vo.save.*;
import com.elitescloud.cloudt.lowcode.dynamic.repo.*;
import com.elitescloud.cloudt.lowcode.dynamic.service.dynamic.DynamicConfigurationMngService;
import com.elitescloud.cloudt.lowcode.dynamic.service.spi.DynamicWorkflowEngine;
import com.elitescloud.cloudt.lowcode.dynamic.service.spi.common.ProcDefInfo;
import com.elitescloud.cloudt.lowcode.dynamic.service.spi.common.WorkflowStatus;
import com.elitescloud.cloudt.system.cacheable.SysCacheMenuRpcService;
import com.elitescloud.cloudt.system.constant.PlatformAppMenusTypeEnum;
import com.elitescloud.cloudt.system.constant.PlatformMenusNodeEnum;
import com.elitescloud.cloudt.system.constant.PlatformMenusOuterLinkTypeEnum;
import com.elitescloud.cloudt.system.provider.SysMenuRpcService;
import com.elitescloud.cloudt.system.provider.dto.save.SysMenuSaveDTO;
import com.google.common.base.Functions;
import org.apache.logging.log4j.util.PropertySource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/7/22
 */
@Service
public class DynamicConfigurationMngServiceImpl extends BaseServiceImpl implements DynamicConfigurationMngService {
    private static final Logger logger = LoggerFactory.getLogger(DynamicConfigurationMngServiceImpl.class);

    private static final Pattern PATTERN_DB_NAME = Pattern.compile("^[a-zA-Z0-9_]+$");

    @Autowired
    private DynamicConfigurationRepoProc repoProc;
    @Autowired
    private DynamicBoModelRepoProc boModelRepoProc;
    @Autowired
    private DynamicConfigurationBoModelRelationRepoProc modelRelationRepoProc;
    @Autowired
    private DynamicBoFieldDefinitionRepoProc boFieldRepoProc;
    @Autowired
    private DynamicDeployRecordRepoProc deployRecordRepoProc;
    @Autowired
    private DynamicDeployModelRecordRepoProc deployModelRecordRepoProc;

    @Autowired
    private PlatformAppProvider appProvider;
    @Autowired
    private SysCacheMenuRpcService cacheMenuRpcService;
    @Autowired
    private SysMenuRpcService menuRpcService;
    @Autowired
    private List<DynamicWorkflowEngine> workflowEngines;
    @Autowired
    private DynamicBoModelDeployService boModelDeployService;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveDynamicConfigurator(DynamicConfiguratorSaveVO saveVO) {
        DynamicConfiguratorDO configuratorDO = null;
        try {
            configuratorDO = this.checkAndConvert(saveVO);
        } catch (Exception e) {
            logger.error("保存功能模块校验失败：", e);
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        repoProc.save(configuratorDO);
        return ApiResult.ok(configuratorDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> publishDynamicConfigurator(DynamicConfiguratorPublishSaveVO saveVO) {
        Assert.notNull(saveVO.getId(), "功能模块ID为空");
        Assert.notBlank(saveVO.getAppCode(), "应用编码为空");
        Assert.notBlank(saveVO.getMenuCode(), "菜单编码为空");

        // 更新为已发布状态
        if (Boolean.TRUE.equals(repoProc.isPublished(saveVO.getId()))) {
            return ApiResult.fail("请先取消发布");
        }
        repoProc.updatePublished(saveVO.getId(), saveVO.getAppCode(), saveVO.getMenuCode());

        // 新增菜单
        this.publishMenu(saveVO);

        return ApiResult.ok(saveVO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> revokePublishDynamicConfigurator(Long id) {
        Assert.notNull(id, "功能模块ID为空");

        var menuCode = repoProc.getMenuCode(id);
        if (CharSequenceUtil.isBlank(menuCode)) {
            return ApiResult.ok(id);
        }

        // 取消发布
        repoProc.revokePublish(id);

        // 删除菜单
        this.revokeMenu(menuCode);

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> deleteDynamicConfigurator(Long id) {
        if (Boolean.TRUE.equals(repoProc.isPublished(id))) {
            return ApiResult.fail("请先取消发布");
        }
        if (Boolean.TRUE.equals(repoProc.isDeployed(id))) {
            return ApiResult.fail("请先取消部署");
        }

        var modelCodes = modelRelationRepoProc.getModelCodes(id);

        // 删除关联
        modelRelationRepoProc.deleteRelation(id, null);
        // 删除模型
        boModelRepoProc.deleteByBoModelCodes(modelCodes);
        // 删除部署记录
        deployRecordRepoProc.deleteByDynamicConfigurator(id);
        deployModelRecordRepoProc.deleteByDynamicConfigurator(id);

        // 删除功能模块
        repoProc.delete(id);
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveBoModel(DynamicBoModelSaveVO saveVO) {
        DynamicBoModelDO boModelDO = null;
        try {
            boModelDO = this.checkAndConvert(saveVO);
        } catch (Exception e) {
            logger.error("保存功能的数据模型时异常：", e);
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        // 保存模型
        boModelRepoProc.save(boModelDO);

        // 保存模型关系
        saveModelRelation(saveVO);

        return ApiResult.ok(boModelDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> deleteBoModel(Long id, Long boModelId, Boolean deleteModel) {
        Assert.notNull(id, "功能模块ID为空");
        Assert.notNull(boModelId, "模型ID为空");
        if (deleteModel == null) {
            deleteModel = true;
        }

        // 获取模型编码
        var boModel = boModelRepoProc.getSimple(boModelId);
        if (boModel == null) {
            return ApiResult.ok(id);
        }
        if (Boolean.TRUE.equals(boModel.getCreationTable())) {
            return ApiResult.fail("表已创建，请先取消部署");
        }

        // 删除关系
        modelRelationRepoProc.deleteRelation(id, boModel.getBoModelCode());

        // 删除模型
        if (deleteModel) {
            boModelRepoProc.delete(boModelId);
            boFieldRepoProc.deleteByBoModelId(boModelId);
        }

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> addBoModelField(Long boModelId, DynamicBoFieldSaveVO fieldSaveVO) {
        DynamicBoFieldDefinitionDO boFieldDO = null;
        try {
            boFieldDO = this.checkAndConvert(boModelId, fieldSaveVO);
        } catch (Exception e) {
            logger.error("保存数据模型字段异常：", e);
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        boFieldRepoProc.save(boFieldDO);

        return ApiResult.ok(boFieldDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> deleteBoModelField(Long boFieldId) {
        Assert.notNull(boFieldId, "字段ID为空");

        boFieldRepoProc.delete(boFieldId);

        return ApiResult.ok(boFieldId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveBoModelFields(Long boModelId, List<DynamicBoFieldSaveVO> fieldSaveList) {
        Assert.notNull(boModelId, "模型ID为空");

        // 模型校验
        String boModelCode = boModelRepoProc.getCode(boModelId);
        Assert.notBlank(boModelCode, "数据模型不存在");

        List<DynamicBoFieldDefinitionDO> boFieldDOList = null;
        try {
            boFieldDOList = this.checkAndConvert(boModelCode, fieldSaveList);
        } catch (Exception e) {
            logger.error("保存数据模型字段异常：", e);
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        // 获取当前的
        var existsIds = boFieldRepoProc.getIdsByBoModelCode(boModelCode);

        if (CollUtil.isEmpty(boFieldDOList)) {
            // 需要清空
            if (!existsIds.isEmpty()) {
                boFieldRepoProc.delete(existsIds);
            }

            return ApiResult.ok(boModelId);
        }

        // 保存新的
        if (!existsIds.isEmpty()) {
            // 清空没的
            var newIds = boFieldDOList.stream().map(DynamicBoFieldDefinitionDO::getId).collect(Collectors.toList());
            var toDelIds = existsIds.stream().filter(t -> !newIds.contains(t)).collect(Collectors.toList());
            if (!toDelIds.isEmpty()) {
                boFieldRepoProc.delete(toDelIds);
            }
        }
        boFieldRepoProc.save(boFieldDOList);

        return ApiResult.ok(boModelId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveFormCfg(DynamicConfiguratorFormSaveVO formSaveVO) {
        Assert.notNull(formSaveVO.getId(), "功能模块ID为空");

        Map<String, Object> formCfg = CollUtil.isEmpty(formSaveVO.getFormJson()) ? null : formSaveVO.getFormJson();
        repoProc.updateFormCfg(formSaveVO.getId(), formCfg);

        return ApiResult.ok(formSaveVO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveListPageCfg(DynamicConfiguratorListPageSaveVO listPageSaveVO) {
        Assert.notNull(listPageSaveVO.getId(), "功能模块ID为空");

        Map<String, Object> listPageCfg = CollUtil.isEmpty(listPageSaveVO.getListPageJson()) ? null : listPageSaveVO.getListPageJson();
        repoProc.updateListPageCfg(listPageSaveVO.getId(), listPageCfg);

        return ApiResult.ok(listPageSaveVO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveApproval(DynamicConfiguratorApprovalSaveVO saveVO) {
        Assert.notNull(saveVO.getConfiguratorId(), "功能模块ID为空");
        Assert.notBlank(saveVO.getApprovalType(), "流程审批类型为空");

        // 获取现有配置
        var existsCfg = repoProc.getWorkflowCfgSimpleBO(saveVO.getConfiguratorId());
        Assert.notNull(existsCfg, "功能模块不存在");

        // 先更新配置
        String approvalKey = null;
        if (saveVO.getProcDefInfo() != null) {
            approvalKey = saveVO.getProcDefInfo().getKey();
            saveVO.getProcDefInfo().setForm(true);
            saveVO.getProcDefInfo().setModuleCode(repoProc.getCode(saveVO.getConfiguratorId()));
            saveVO.getProcDefInfo().setModuleName(repoProc.getName(saveVO.getConfiguratorId()));
        }
        boolean created = !ApproveTypeUdc.EMPTY.getValue().equals(saveVO.getApprovalType());
        repoProc.updateApprovalCfg(saveVO.getConfiguratorId(), created, saveVO.getApprovalType(), saveVO.getApprovalEngine(),
                approvalKey, saveVO.getApprovalJson());

        // 调用流程引擎保存流程
        if (StringUtils.hasText(saveVO.getApprovalEngine())) {
            boolean matched = false;
            for (DynamicWorkflowEngine workflowEngine : workflowEngines) {
                if (saveVO.getApprovalEngine().equals(workflowEngine.engineType())) {
                    matched = true;
                    workflowEngine.createProcDef(saveVO.getProcDefInfo());
                }
            }
            Assert.isTrue(matched, "保存流程失败");
        }

        return ApiResult.ok(saveVO.getConfiguratorId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> deployApproval(Long id) {
        Assert.notNull(id, "功能模块ID为空");

        var bo = repoProc.getWorkflowCfgSimpleBO(id);
        Assert.notNull(bo, "功能模块不存在");
        Assert.isTrue(Boolean.TRUE.equals(bo.getCreatedApproval()), "未创建工作流");
        Assert.notBlank(bo.getApprovalEngine(), "请先选择工作流引擎");

        // 更新部署状态
        repoProc.updateDeployedApproval(id, true);

        var matched = false;
        for (DynamicWorkflowEngine workflowEngine : workflowEngines) {
            if (workflowEngine.engineType().equals(bo.getApprovalEngine())) {
                workflowEngine.deploy(bo.getApprovalKey());
                matched = true;
                break;
            }
        }
        if (!matched) {
            throw new BusinessException("未找到工作流引擎" + bo.getApprovalEngine());
        }
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> deleteApproval(Long id) {
        Assert.notNull(id, "功能模块ID为空");

        var bo = repoProc.getWorkflowCfgSimpleBO(id);
        Assert.notNull(bo, "功能模块不存在");
        Assert.isTrue(Boolean.TRUE.equals(bo.getCreatedApproval()), "未创建工作流");
        Assert.notBlank(bo.getApprovalEngine(), "请先选择工作流引擎");

        var published = repoProc.getPublished(id);
        Assert.isTrue(Boolean.FALSE.equals(published), "功能已发布，不可删除流程");

        repoProc.deleteApproval(id);

        var matched = false;
        for (DynamicWorkflowEngine workflowEngine : workflowEngines) {
            if (workflowEngine.engineType().equals(bo.getApprovalEngine())) {
                workflowEngine.deleteProcDef(bo.getApprovalKey());
                matched = true;
                break;
            }
        }
        if (!matched) {
            throw new BusinessException("未找到工作流引擎" + bo.getApprovalEngine());
        }
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> deploy(Long id) {
        Assert.notNull(id, "功能模块ID为空");

        this.deploy(id, DeployType.DEPLOY);

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> redeploy(Long id) {
        Assert.notNull(id, "功能模块ID为空");

        this.deploy(id, DeployType.REDEPLOY);

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> cancelDeploy(Long id) {
        Assert.notNull(id, "功能模块ID为空");

        this.deploy(id, DeployType.CANCEL_DEPLOY);

        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<DynamicConfiguratorDetailRespVO> getDynamicConfigurator(Long id) {
        Assert.notNull(id, "功能模块ID为空");
        var configuratorDO = repoProc.get(id);
        if (configuratorDO == null) {
            return ApiResult.fail("数据不存在");
        }

        var respVO = DynamicConfiguratorConvert.INSTANCE.do2DetailRespVO(configuratorDO);
        respVO.setApprovalTypeName(super.udcValue(new ApproveTypeUdc(ApproveTypeUdc.UDC_CODE, respVO.getApprovalType())));

        if (StringUtils.hasText(respVO.getAppCode())) {
            var appMap = appProvider.all().stream().collect(Collectors.toMap(CodeNameParam::getCode, CodeNameParam::getName, (t1, t2) -> t1));
            respVO.setAppName(appMap.get(respVO.getAppCode()));
        }

        // 发布的菜单信息
        if (StringUtils.hasText(respVO.getMenuCode())) {
            var menus = cacheMenuRpcService.getMenuWithParents(respVO.getMenuCode());
            if (CollUtil.isNotEmpty(menus)) {
                respVO.setMenus(menus);
                respVO.setMenuName(menus.get(menus.size() - 1).getName());
            }
        }
        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<PagingVO<DynamicConfiguratorPageRespVO>> pageQuery(DynamicConfiguratorPageQueryVO queryVO) {
        var pageData = repoProc.pageMng(queryVO);
        if (pageData.isEmpty()) {
            return ApiResult.ok(pageData);
        }

        var approvalTypeMap = super.udcMap(ApproveTypeUdc.EMPTY);
        var appMap = appProvider.all().stream().collect(Collectors.toMap(CodeNameParam::getCode, CodeNameParam::getName, (t1, t2) -> t1));

        pageData.each(t -> {
            t.setApprovalTypeName(approvalTypeMap.get(t.getApprovalType()));
            t.setAppName(appMap.get(t.getAppCode()));

            if (StringUtils.hasText(t.getMenuCode())) {
                var menus = cacheMenuRpcService.getMenuWithParents(t.getMenuCode());
                if (CollUtil.isNotEmpty(menus)) {
                    t.setMenus(menus);
                    t.setMenuName(menus.get(menus.size() - 1).getName());
                }
            }
        });

        return ApiResult.ok(pageData);
    }

    @Override
    public ApiResult<List<DynamicConfiguratorBoModelListRespVO>> listBoModel(Long id) {
        Assert.notNull(id, "功能模块ID为空");

        var respVoList = modelRelationRepoProc.listBoModelRelation(id);
        return ApiResult.ok(respVoList);
    }

    @Override
    public ApiResult<List<DynamicConfiguratorBoModelDetailListRespVO>> listBoModelDetail(Long id) {
        Assert.notNull(id, "功能模块ID为空");

        var respVoList = modelRelationRepoProc.listBoModelDetail(id);
        if (CollUtil.isEmpty(respVoList)) {
            return ApiResult.ok(respVoList);
        }

        // 获取字段列表
        var fieldsMap = boFieldRepoProc.listByConfiguratorId(id).stream().
        map(t -> {
            var respVO = DynamicConfiguratorConvert.INSTANCE.do2DetailRespVO(t);
            respVO.setBoFieldId(t.getId());
            respVO.setBoModelCode(t.getBasicModuleCode());
            return respVO;
        }).sorted(Comparator.comparing(DynamicConfiguratorBoFieldDetailRespVO::getBasicDisplayOrder, Comparator.nullsLast(Integer::compareTo)))
                .collect(Collectors.groupingBy(DynamicConfiguratorBoFieldDetailRespVO::getBoModelCode));
        for (DynamicConfiguratorBoModelDetailListRespVO respVO : respVoList) {
            respVO.setFieldList(fieldsMap.getOrDefault(respVO.getBoModelCode(), Collections.emptyList()));
        }
        return ApiResult.ok(respVoList);
    }

    @Override
    public ApiResult<DynamicConfiguratorBoModelDetailRespVO> getBoModel(Long id, Long boModelId) {
        Assert.notNull(id, "功能模块ID为空");
        Assert.notNull(boModelId, "数据模型ID为空");

        // 获取模型
        var modelBO = boModelRepoProc.getSimple(boModelId);
        if (modelBO == null) {
            return ApiResult.fail("数据模型不存在");
        }

        // 查询关系
        var modelRelationDO = modelRelationRepoProc.getBoModelRelation(id, modelBO.getBoModelCode());
        if (modelRelationDO == null) {
            return ApiResult.fail("数据模型关系不存在");
        }

        DynamicConfiguratorBoModelDetailRespVO respVO = new DynamicConfiguratorBoModelDetailRespVO();
        respVO.setConfiguratorId(id);
        respVO.setBoModelCode(modelBO.getBoModelCode());
        respVO.setBoModeName(modelBO.getBoModeName());
        respVO.setBoModelType(modelRelationDO.getBoModelType());
        respVO.setBoModelTypeName(modelBO.getDescription());
        respVO.setDescription(modelRelationDO.getBoModelCode());

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<List<DynamicConfiguratorBoFieldListRespVO>> getBoFieldList(Long boModelId) {
        Assert.notNull(boModelId, "数据模型ID为空");

        var respVoList = boFieldRepoProc.listByBoModelId(boModelId)
                .stream()
                .sorted(Comparator.comparing(DynamicBoFieldDefinitionDO::getBasicDisplayOrder, Comparator.nullsLast(Integer::compareTo)))
                .map(t -> {
                    var respVO = DynamicConfiguratorConvert.INSTANCE.do2ListRespVO(t);
                    respVO.setBoFieldId(t.getId());

                    return respVO;
                }).collect(Collectors.toList());
        return ApiResult.ok(respVoList);
    }

    @Override
    public ApiResult<DynamicConfiguratorBoFieldDetailRespVO> getBoFieldDetail(Long boFieldId) {
        Assert.notNull(boFieldId, "字段ID为空");

        var fieldDO = boFieldRepoProc.get(boFieldId);
        if (fieldDO == null) {
            return ApiResult.fail("数据不存在");
        }

        var respVO = DynamicConfiguratorConvert.INSTANCE.do2DetailRespVO(fieldDO);
        respVO.setBoFieldId(fieldDO.getId());
        respVO.setBoModelCode(fieldDO.getBasicModuleCode());

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<Map<String, Object>> getFormCfg(Long id) {
        Assert.notNull(id, "功能模块ID为空");

        var formCfg = repoProc.getFormJson(id);
        return ApiResult.ok(ObjUtil.defaultIfNull(formCfg, Collections.emptyMap()));
    }

    @Override
    public ApiResult<Map<String, Object>> getListPageCfg(Long id) {
        Assert.notNull(id, "功能模块ID为空");

        var listPageCfg = repoProc.getListPageJson(id);
        return ApiResult.ok(ObjUtil.defaultIfNull(listPageCfg, Collections.emptyMap()));
    }

    @Override
    public ApiResult<DynamicConfiguratorApprovalRespVO> getApprovalInfo(Long id) {
        Assert.notNull(id, "功能模块ID为空");

        // 获取流程配置
        var cfgBO = repoProc.getWorkflowCfgSimpleBO(id);
        Assert.notNull(cfgBO, "功能模块不存在");

        DynamicConfiguratorApprovalRespVO respVO = new DynamicConfiguratorApprovalRespVO();
        respVO.setCreatedApproval(Boolean.TRUE.equals(cfgBO.getCreatedApproval()));
        respVO.setApprovalType(cfgBO.getApprovalType());
        if (StringUtils.hasText(cfgBO.getApprovalType())) {
            respVO.setApprovalTypeName(super.udcValue(new ApproveTypeUdc(cfgBO.getApprovalType())));
        }

        // 流程引擎
        if (StringUtils.hasText(cfgBO.getApprovalEngine())) {
            respVO.setApprovalEngine(cfgBO.getApprovalEngine());
            boolean engineMatched = false;
            for (DynamicWorkflowEngine workflowEngine : workflowEngines) {
                if (cfgBO.getApprovalEngine().equals(workflowEngine.engineType())) {
                    engineMatched = true;
                    respVO.setApprovalEngineName(workflowEngine.engineName());

                    if (StringUtils.hasText(cfgBO.getApprovalKey())) {
                        try {
                            respVO.setProcDefInfo(workflowEngine.getProcDefInfo(cfgBO.getApprovalKey()));
                            respVO.setEngineState(WorkflowEngineStatusUdc.RUNNING.getValue());
                            respVO.setEngineStateName(super.udcValue(new WorkflowEngineStatusUdc(WorkflowEngineStatusUdc.RUNNING.getValue())));
                        } catch (Exception e) {
                            logger.error("获取流程定义异常：{}", cfgBO.getApprovalKey(), e);
                            respVO.setEngineState(WorkflowEngineStatusUdc.CONNECTION_EXP.getValue());
                            respVO.setEngineStateName(super.udcValue(new WorkflowEngineStatusUdc(WorkflowEngineStatusUdc.CONNECTION_EXP.getValue())));
                            respVO.setEngineExpMsg(e.getMessage());
                        }
                    }
                }
            }
            Assert.isTrue(engineMatched, "未知流程引擎类型" + cfgBO.getApprovalEngine());
        }
        respVO.setApprovalKey(cfgBO.getApprovalKey());
        respVO.setApprovalJson(cfgBO.getApprovalJson());
        if (respVO.getProcDefInfo() == null) {
            respVO.setCreatedApproval(false);
            respVO.setProcDefInfo(new ProcDefInfo());
            respVO.getProcDefInfo().setAppPrefixUrl("http://" + Application.NAME);
        } else {
            respVO.setCreatedApproval(true);
        }
        if (CharSequenceUtil.isBlank(respVO.getProcDefInfo().getModuleCode())) {
            respVO.getProcDefInfo().setModuleCode(repoProc.getCode(id));
        }
        if (CharSequenceUtil.isBlank(respVO.getProcDefInfo().getModuleName())) {
            respVO.getProcDefInfo().setModuleName(repoProc.getName(id));
        }
        if (CharSequenceUtil.isBlank(respVO.getProcDefInfo().getName())) {
            respVO.getProcDefInfo().setName(respVO.getProcDefInfo().getName() + "审批");
        }

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<List<CodeNameParam>> getApprovalEngineList() {
        var engineList = workflowEngines.stream().map(t -> new CodeNameParam(t.engineType(), t.engineName())).collect(Collectors.toList());
        return ApiResult.ok(engineList);
    }

    @Override
    public ApiResult<WorkflowStatus> getApprovalEngineStatus(String engineType) {
        Assert.notBlank(engineType, "工作流引擎类型为空");

        for (DynamicWorkflowEngine workflowEngine : workflowEngines) {
            if (engineType.equals(workflowEngine.engineType())) {
                return ApiResult.ok(workflowEngine.getEngineStatus());
            }
        }

        return ApiResult.fail("未知工作流引擎类型");
    }

    @Override
    public ApiResult<DynamicDeployInfoRespVO> getDeployInfo(Long id) {
        Assert.notNull(id, "功能模块ID为空");

        DynamicDeployInfoRespVO respVO = new DynamicDeployInfoRespVO();
        respVO.setDeployed(repoProc.getDeploy(id));

        // 查询执行记录
        var modelRecordsMap = deployModelRecordRepoProc.listByConfigurator(id).stream()
                        .collect(Collectors.groupingBy(DynamicDeployModelRecordDO::getRecordId));
        var recordList = deployRecordRepoProc.listByDynamicConfigurator(id).stream()
                .sorted(Comparator.comparing(DynamicDeployRecordDO::getStartTime).reversed())
                .map(t -> {
                    DynamicDeployRecordRespVO recordRespVO = new DynamicDeployRecordRespVO();
                    recordRespVO.setId(t.getId());
                    recordRespVO.setDeployType(t.getDeployType());
                    recordRespVO.setStartTime(t.getStartTime());
                    recordRespVO.setFinishTime(t.getFinishTime());
                    recordRespVO.setSuccess(t.getSuccess());

                    if (t.getFinished() == null || !t.getFinished()) {
                        recordRespVO.setResult("进行中");
                    } else {
                        recordRespVO.setResult(Boolean.TRUE.equals(t.getSuccess()) ? "成功" : "失败");
                    }
                    recordRespVO.setCreator(t.getCreator());

                    recordRespVO.setLog(modelRecordsMap.getOrDefault(t.getId(), Collections.emptyList()).stream().map(DynamicDeployModelRecordDO::getPrettyLogJson).collect(Collectors.joining(";")));

                    return recordRespVO;
                }).collect(Collectors.toList());
        respVO.setRecordList(recordList);

        return ApiResult.ok(respVO);
    }

    private void deploy(long id, DeployType deployType) {
        var configuratorCode = repoProc.getCode(id);
        Assert.notBlank(configuratorCode, "功能模块不存在");
        var tableList = this.queryTablesByDynamicCode(configuratorCode);
        Assert.notEmpty(tableList, "请先配置数据模型");

        boModelDeployService.deploy(deployType, tableList, configuratorCode);
    }

    private List<DynamicModelTableBO> queryTablesByDynamicCode(String dynamicConfiguratorCode) {
        var tableList = modelRelationRepoProc.listModelTableByConfiguratorCode(dynamicConfiguratorCode);
        if (tableList.isEmpty()) {
            return tableList;
        }

        var boModelCodes = tableList.stream().map(DynamicModelTableBO::getBoModelCode).collect(Collectors.toSet());
        var columnsMap = boFieldRepoProc.listColumnBO(boModelCodes).stream().collect(Collectors.groupingBy(DynamicModelColumnBO::getBasicModuleCode));
        for (DynamicModelTableBO tableBO : tableList) {
            if (!columnsMap.containsKey(tableBO.getBoModelCode())) {
                continue;
            }

            var columnList = columnsMap.get(tableBO.getBoModelCode())
                    .stream()
                    .sorted(Comparator.comparing(DynamicModelColumnBO::getBasicDisplayOrder, Comparator.nullsLast(Integer::compareTo)))
                    .collect(Collectors.toList());
            tableBO.setColumnList(columnList);

            // 添加基础字段
            this.addBaseColumns(columnList);
        }

        return tableList.stream().filter(t -> CollUtil.isNotEmpty(t.getColumnList())).collect(Collectors.toList());
    }

    private void addBaseColumns(List<DynamicModelColumnBO> columns) {
        var columnsMap = columns.stream().collect(Collectors.toMap(DynamicModelColumnBO::getBasicKey, Functions.identity(), (t1, t2) -> t1));
        var baseColumns = this.getBaseColumns();
        for (DynamicModelColumnBO baseColumn : baseColumns) {
            if (columnsMap.containsKey(baseColumn.getBasicKey())) {
                continue;
            }
            if ("id".equals(baseColumn.getBasicKey())) {
                columns.add(0, baseColumn);
                continue;
            }
            columns.add(baseColumn);
        }
    }

    private List<DynamicModelColumnBO> getBaseColumns() {
        List<DynamicModelColumnBO> columns = new ArrayList<>(16);
        DynamicModelColumnBO id = new DynamicModelColumnBO();
        id.setBasicKey("id");
        id.setBasicName("主键");
        id.setBasicType(BoModelFieldTypeEnum.LONG);
        id.setPrimaryKey(true);
        id.setDbFieldNullable(false);
        id.setDbFieldUnique(true);
        id.setDbFieldIndex(false);
        columns.add(id);

        DynamicModelColumnBO tenantId = new DynamicModelColumnBO();
        tenantId.setBasicKey("tenant_id");
        tenantId.setBasicName("租户ID");
        tenantId.setBasicType(BoModelFieldTypeEnum.LONG);
        tenantId.setBasicDefaultValue(TenantConstant.DEFAULT_TENANT_ID.toString());
        tenantId.setDbFieldNullable(false);
        columns.add(tenantId);

        DynamicModelColumnBO belongOrgId = new DynamicModelColumnBO();
        belongOrgId.setBasicKey("belong_org_id");
        belongOrgId.setBasicName("所属组织ID");
        belongOrgId.setBasicType(BoModelFieldTypeEnum.LONG);
        columns.add(belongOrgId);

        DynamicModelColumnBO tenantOrgId = new DynamicModelColumnBO();
        tenantOrgId.setBasicKey("tenant_org_id");
        tenantOrgId.setBasicName("租户组织ID");
        tenantOrgId.setBasicType(BoModelFieldTypeEnum.LONG);
        columns.add(tenantOrgId);

        DynamicModelColumnBO remark = new DynamicModelColumnBO();
        remark.setBasicKey("remark");
        remark.setBasicName("备注");
        remark.setBasicType(BoModelFieldTypeEnum.STRING);
        columns.add(remark);

        DynamicModelColumnBO createUserId = new DynamicModelColumnBO();
        createUserId.setBasicKey("create_user_id");
        createUserId.setBasicName("记录创建者");
        createUserId.setBasicType(BoModelFieldTypeEnum.LONG);
        columns.add(createUserId);

        DynamicModelColumnBO creator = new DynamicModelColumnBO();
        creator.setBasicKey("creator");
        creator.setBasicName("记录创建者");
        creator.setBasicType(BoModelFieldTypeEnum.STRING);
        columns.add(creator);

        DynamicModelColumnBO createTime = new DynamicModelColumnBO();
        createTime.setBasicKey("create_time");
        createTime.setBasicName("记录创建时间");
        createTime.setBasicType(BoModelFieldTypeEnum.DATE_TIME);
        columns.add(createTime);

        DynamicModelColumnBO modifyUserId = new DynamicModelColumnBO();
        modifyUserId.setBasicKey("modify_user_id");
        modifyUserId.setBasicName("记录最后更新者ID");
        modifyUserId.setBasicType(BoModelFieldTypeEnum.LONG);
        columns.add(modifyUserId);

        DynamicModelColumnBO updater = new DynamicModelColumnBO();
        updater.setBasicKey("updater");
        updater.setBasicName("记录最后更新者");
        updater.setBasicType(BoModelFieldTypeEnum.STRING);
        columns.add(updater);

        DynamicModelColumnBO modifyTime = new DynamicModelColumnBO();
        modifyTime.setBasicKey("modify_time");
        modifyTime.setBasicName("记录最后更新时间");
        modifyTime.setBasicType(BoModelFieldTypeEnum.DATE_TIME);
        columns.add(modifyTime);

        DynamicModelColumnBO deleteFlag = new DynamicModelColumnBO();
        deleteFlag.setBasicKey("delete_flag");
        deleteFlag.setBasicName("删除标记");
        deleteFlag.setBasicType(BoModelFieldTypeEnum.INTEGER);
        deleteFlag.setBasicDefaultValue("0");
        deleteFlag.setDbFieldNullable(false);
        columns.add(deleteFlag);

        DynamicModelColumnBO auditDataVersion = new DynamicModelColumnBO();
        auditDataVersion.setBasicKey("audit_data_version");
        auditDataVersion.setBasicName("锁版本");
        auditDataVersion.setBasicType(BoModelFieldTypeEnum.INTEGER);
        auditDataVersion.setBasicDefaultValue("0");
        auditDataVersion.setDbFieldNullable(false);
        columns.add(auditDataVersion);

        DynamicModelColumnBO secBuId = new DynamicModelColumnBO();
        secBuId.setBasicKey("sec_bu_Id");
        secBuId.setBasicName("数据归属组织id");
        secBuId.setBasicType(BoModelFieldTypeEnum.LONG);
        columns.add(secBuId);

        DynamicModelColumnBO secUserId = new DynamicModelColumnBO();
        secUserId.setBasicKey("sec_user_id");
        secUserId.setBasicName("数据归属雇员id");
        secUserId.setBasicType(BoModelFieldTypeEnum.LONG);
        columns.add(secUserId);

        DynamicModelColumnBO secOuId = new DynamicModelColumnBO();
        secOuId.setBasicKey("sec_ou_id");
        secOuId.setBasicName("数据归属公司id");
        secOuId.setBasicType(BoModelFieldTypeEnum.LONG);
        columns.add(secOuId);

        return columns;
    }

    private DynamicConfiguratorDO checkAndConvert(DynamicConfiguratorSaveVO saveVO) {
        DynamicConfiguratorDO configuratorDO = saveVO.getId() == null ? new DynamicConfiguratorDO() : repoProc.get(saveVO.getId());
        Assert.notNull(configuratorDO, "修改的数据不存在");

        Assert.notBlank(saveVO.getDynamicConfiguratorCode(), "编码为空");
        if (saveVO.getId() != null) {
            Assert.isTrue(saveVO.getDynamicConfiguratorCode().equals(configuratorDO.getDynamicConfiguratorCode()), "编码不可修改");
        }
        var exists = repoProc.existsCode(saveVO.getDynamicConfiguratorCode(), saveVO.getId());
        Assert.isFalse(exists, "编码已存在");

        DynamicConfiguratorConvert.INSTANCE.saveVo2DO(saveVO, configuratorDO);

        // 默认值设置
        if (CharSequenceUtil.isBlank(saveVO.getApprovalType())) {
            configuratorDO.setApprovalType(ApproveTypeUdc.EMPTY.getValue());
        }
        ObjUtil.ifNull(configuratorDO.getApprovalKey(), "", configuratorDO::setApprovalKey);

        return configuratorDO;
    }

    private DynamicBoModelDO checkAndConvert(DynamicBoModelSaveVO saveVO) {
        DynamicBoModelDO boModelDO = saveVO.getBoModelId() == null ? new DynamicBoModelDO() : boModelRepoProc.get(saveVO.getBoModelId());
        Assert.notNull(boModelDO, "修改的数据不存在");

        boolean exists = false;
        // 模型编码
        Assert.notBlank(saveVO.getBoModelCode(), "模型编码为空");
        Assert.isTrue(validateDbName(saveVO.getBoModelCode()), "模型编码只能是由字母、下划线和数字组成，且长度不超过20");
        if (!saveVO.getBoModelCode().equalsIgnoreCase(boModelDO.getBoModelCode())) {
            if (saveVO.getBoModelId() != null) {
                throw new BusinessException("编码不可修改");
            }
            exists = tenantDataIsolateProvider.byNone(() -> boModelRepoProc.existsCode(saveVO.getBoModelCode(), saveVO.getBoModelId()));
            Assert.isFalse(exists, "编码已存在");
        }

        DynamicConfiguratorConvert.INSTANCE.copyVo2DO(saveVO, boModelDO);

        // 默认值设置
        if (boModelDO.getCreationTable() == null) {
            boModelDO.setCreationTable(false);
        }
        if (!boModelDO.getCreationTable()) {
            boModelDO.setDatabaseTableName(DynamicConstants.MODEL_TABLE_PREFIX + boModelDO.getBoModelCode());
        }
        return boModelDO;
    }

    private DynamicBoFieldDefinitionDO checkAndConvert(Long boModelId, DynamicBoFieldSaveVO saveVO) {
        Assert.notNull(boModelId, "模型ID为空");
        DynamicBoFieldDefinitionDO boFieldDO = saveVO.getBoFieldId() == null ? new DynamicBoFieldDefinitionDO() :
                boFieldRepoProc.get(saveVO.getBoFieldId());
        Assert.notNull(boFieldDO, "修改的数据不存在");

        // 模型校验
        String boModelCode = StringUtils.hasText(boFieldDO.getBasicModuleCode()) ? boFieldDO.getBasicModuleCode() :
                boModelRepoProc.getCode(boModelId);
        Assert.notBlank(boModelCode, "数据模型不存在");

        // 字段标识
        Assert.notBlank(saveVO.getBasicKey(), "字段标识为空");
        Assert.isTrue(this.validateDbName(saveVO.getBasicKey()), "字段标识只能由字母、数字和下划线组成，且长度不超过20");
        if (StringUtils.hasText(saveVO.getBasicKeyAs()) && !validateDbName(saveVO.getBasicKeyAs())) {
            throw new IllegalArgumentException("字段标识别名只能由字母、数字和下划线组成，且长度不超过20");
        }
        if (!saveVO.getBasicKey().equals(boFieldDO.getBasicKey())) {
            var basicKeys = boFieldRepoProc.getBasicKeys(boModelCode);
            Assert.isFalse(basicKeys.contains(saveVO.getBasicKey()), "字段标识已存在");
        }

        Assert.notNull(saveVO.getBasicType(), "字段类型为空");

        // 小数校验
        if (saveVO.getDbFieldPrecision() != null && saveVO.getDbFieldScale() != null) {
            Assert.isTrue(saveVO.getDbFieldPrecision() >= saveVO.getDbFieldScale(), "小数部分位数不能小于小数总位数");
        }

        DynamicConfiguratorConvert.INSTANCE.copyVo2DO(saveVO, boFieldDO);

        // 默认值
        boFieldDO.setBasicModuleCode(boModelCode);
        if (boFieldDO.getBasicDisplayOrder() == null) {
            boFieldDO.setBasicDisplayOrder(ObjUtil.defaultIfNull(boFieldRepoProc.getMaxSortNo(boModelCode), 0) + 1);
        }
        ObjUtil.ifNull(saveVO.getDbField(), true, boFieldDO::setDbField);
        ObjUtil.ifNull(saveVO.getBasicState(), true, boFieldDO::setBasicState);

        return boFieldDO;
    }

    private List<DynamicBoFieldDefinitionDO> checkAndConvert(String boModelCode, List<DynamicBoFieldSaveVO> saveVoList) {
        Assert.notBlank(boModelCode, "模型编码为空");
        if (CollUtil.isEmpty(saveVoList)) {
            return Collections.emptyList();
        }

        // 查询老数据
        var existIds = saveVoList.stream().map(DynamicBoFieldSaveVO::getBoFieldId).filter(Objects::nonNull).collect(Collectors.toList());
        Map<Long, DynamicBoFieldDefinitionDO> existsFieldDoMap = existIds.isEmpty() ? Collections.emptyMap() :
                boFieldRepoProc.get(existIds).stream().collect(Collectors.toMap(DynamicBoFieldDefinitionDO::getId, Functions.identity(), (t1, t2) -> t1));

        // 转为do
        List<DynamicBoFieldDefinitionDO> fieldDoList = new ArrayList<>(saveVoList.size());
        Set<String> existsBasicKeys = new HashSet<>(saveVoList.size());
        DynamicBoFieldDefinitionDO fieldDO = null;
        int index = 0;
        for (DynamicBoFieldSaveVO saveVO : saveVoList) {
            fieldDO = saveVO.getBoFieldId() == null ? new DynamicBoFieldDefinitionDO() : existsFieldDoMap.get(saveVO.getBoFieldId());
            Assert.notNull(fieldDO, "存在修改的数据不存在");

            // 字段标识
            Assert.notBlank(saveVO.getBasicKey(), "字段标识为空");
            Assert.isTrue(this.validateDbName(saveVO.getBasicKey()), "字段标识只能由字母、数字和下划线组成，且长度不超过20");
            if (StringUtils.hasText(saveVO.getBasicKeyAs()) && !validateDbName(saveVO.getBasicKeyAs())) {
                throw new IllegalArgumentException("字段标识别名只能由字母、数字和下划线组成，且长度不超过20");
            }
            Assert.isFalse(existsBasicKeys.contains(saveVO.getBasicKey()), "字段标识" + saveVO.getBasicKey() + "已存在");
            existsBasicKeys.add(saveVO.getBasicKey());

            Assert.notNull(saveVO.getBasicType(), "字段类型为空");

            // 小数校验
            if (saveVO.getDbFieldPrecision() != null && saveVO.getDbFieldScale() != null) {
                Assert.isTrue(saveVO.getDbFieldPrecision() >= saveVO.getDbFieldScale(), "小数部分位数不能小于小数总位数");
            }

            DynamicConfiguratorConvert.INSTANCE.copyVo2DO(saveVO, fieldDO);

            // 默认值
            fieldDO.setBasicModuleCode(boModelCode);
            fieldDO.setBasicDisplayOrder(index++);
            ObjUtil.ifNull(saveVO.getDbField(), true, fieldDO::setDbField);
            ObjUtil.ifNull(saveVO.getBasicState(), true, fieldDO::setBasicState);

            fieldDoList.add(fieldDO);
        }
        return fieldDoList;
    }

    private Long saveModelRelation(DynamicBoModelSaveVO saveVO) {
        Assert.notNull(saveVO.getConfiguratorId(), "功能模块ID为空");
        var configuratorCode = repoProc.getCode(saveVO.getConfiguratorId());
        Assert.notBlank(configuratorCode, "功能模块不存在");

        // 主表只能有一个
        Assert.notBlank(saveVO.getBoModelType(), "关系类型为空");
        var primaryCodes = modelRelationRepoProc.getPrimaryModelCodes(configuratorCode);
        if (ModelRelationTypeUdc.PRIMARY.getValue().equals(saveVO.getBoModelType())) {
            Assert.isTrue(primaryCodes.isEmpty() || primaryCodes.contains(saveVO.getBoModelCode()), "主表只能有一个");
        }

        DynamicConfiguratorBoModelRelationDO boModelRelationDO = modelRelationRepoProc.getBoModelRelation(configuratorCode, saveVO.getBoModelCode());
        if (boModelRelationDO == null) {
            boModelRelationDO = new DynamicConfiguratorBoModelRelationDO();
            boModelRelationDO.setDynamicConfiguratorCode(configuratorCode);
        }
        boModelRelationDO.setBoModelCode(saveVO.getBoModelCode());
        boModelRelationDO.setBoModelType(saveVO.getBoModelType());

        modelRelationRepoProc.save(boModelRelationDO);
        return boModelRelationDO.getId();
    }

    private void publishMenu(DynamicConfiguratorPublishSaveVO saveVO) {
        SysMenuSaveDTO saveDTO = new SysMenuSaveDTO();
        saveDTO.setMenusAppCode(saveVO.getAppCode());
        saveDTO.setMenusName(saveVO.getMenuName());
        saveDTO.setMenusTypeEnum(PlatformAppMenusTypeEnum.MENUS_TYPE_BUS);
        saveDTO.setMenusCode(saveVO.getMenuCode());
        saveDTO.setMenusParentCode(saveVO.getMenuGroupCode());
        saveDTO.setNodeTypeEnum(PlatformMenusNodeEnum.MENUS);
        saveDTO.setMenusRoute(saveVO.getMenuRoute());
        saveDTO.setMenusIcon(null);
        saveDTO.setOuterLink(ObjUtil.defaultIfNull(saveVO.getOuterLink(), false));
        if (saveDTO.getOuterLink()) {
            saveDTO.setOuterLinkTypeEnum(ObjUtil.defaultIfNull(saveVO.getOuterLinkTypeEnum(), PlatformMenusOuterLinkTypeEnum.URL));
        }

        var saveResult = menuRpcService.addMenu(saveDTO);
        if (!saveResult.isSuccess()) {
            throw new BusinessException("保存菜单失败，" + saveResult.getMsg());
        }
    }

    private void revokeMenu(String menuCode) {
        menuRpcService.removeMenu(menuCode).computeData();
    }

    /**
     * 是否已创建表
     *
     * @param boModelId
     * @return
     */
    private boolean hasTableCreated(Long boModelId) {
        var created = boModelRepoProc.getCode(boModelId);
        return Boolean.TRUE.equals(created);
    }

    private boolean validateDbName(String str) {
        return str.length() <= 20 && PATTERN_DB_NAME.matcher(str).matches();
    }
}
