package com.elitescloud.cloudt.platform.service.impl;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.constant.SysNumType;
import com.elitescloud.cloudt.platform.convert.SysPlatformNumberRuleDtlConvert;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformNumberRuleDtlDO;
import com.elitescloud.cloudt.platform.model.params.seq.SysPlatformNumberRuleDtlDTO;
import com.elitescloud.cloudt.platform.model.params.seq.SysPlatformNumberRuleDtlVO;
import com.elitescloud.cloudt.platform.service.ISysPlatformNumberRuleDtlService;
import com.elitescloud.cloudt.platform.service.repo.number.SysPlatformNumberRuleDtlRepo;
import com.elitescloud.cloudt.platform.service.repo.number.SysPlatformNumberRuleDtlRepoProc;
import com.elitescloud.cloudt.platform.service.repo.number.SysPlatformNumberRuleRepoProc;
import com.elitescloud.cloudt.system.service.SysTenantBasicDataService;
import com.elitescloud.cloudt.system.service.config.TenantProperties;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/6/18
 */
@Service
@Log4j2
public class ISysPlatformNumberRuleDtlServiceImpl extends BaseServiceImpl implements ISysPlatformNumberRuleDtlService {
    private static final SysPlatformNumberRuleDtlConvert CONVERT = SysPlatformNumberRuleDtlConvert.INSTANCE;

    @Autowired
    private SysPlatformNumberRuleDtlRepo sysNumberRuleDtlRepo;
    @Autowired
    private SysPlatformNumberRuleRepoProc numberRuleRepoProc;
    @Autowired
    private SysPlatformNumberRuleDtlRepoProc sysPlatformNumberRuleDtlRepoProc;
    @Autowired(required = false)
    private TenantProperties tenantProperties;
    @Autowired
    private SysTenantBasicDataService tenantBasicDataService;

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> updateInBatch(List<SysPlatformNumberRuleDtlVO> details) {
        if (CollectionUtils.isEmpty(details)) {
            return ApiResult.fail("规则明细为空");
        }
        Long ruleId = details.get(0).getRuleId();
        if (ruleId == null) {
            return ApiResult.fail("发号规则ID为空");
        }
        var rule = numberRuleRepoProc.get(ruleId);
        if (rule == null) {
            return ApiResult.fail("发号规则不存在");
        }

        // 先删除旧的
        sysPlatformNumberRuleDtlRepoProc.deleteByRuleId(ruleId);

        // 保存新的
        AtomicInteger i = new AtomicInteger();
        List<SysPlatformNumberRuleDtlDO> detailDoList = details.stream()
                .map(detail -> {
                    detail.setRuleId(ruleId);
                    detail.setRuleCode(rule.getRuleCode());
                    if (detail.getSeq() == null) {
                        detail.setSeq(i.getAndIncrement());
                    }

                    // 如果是下一编号，则宽度不能小于0
                    if (SysNumType.NN.name().equals(detail.getNumberType())){
                        Assert.isTrue(detail.getNnLen() != null && detail.getNnLen() > 0, "取号类型是[下一编号]时自增序号宽度不能小于0");
                    }

                    return CONVERT.voToDO(detail);
                }).collect(Collectors.toList());

        sysNumberRuleDtlRepo.saveAll(detailDoList);

        // 同步给租户
        this.syncToTenant(rule.getAppCode(), rule.getRuleCode());

        return ApiResult.ok(ruleId);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Boolean> removeByIds(List<Long> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return ApiResult.fail("ID为空");
        }
        Long ruleId = null;
        for (Long id : ids) {
            ruleId = sysPlatformNumberRuleDtlRepoProc.getRuleId(id);
            if (ruleId != null) {
                break;
            }
        }
        if (ruleId == null) {
            // 都不存在
            return ApiResult.fail("记录不存在");
        }
        var rule = numberRuleRepoProc.get(ruleId);
        if (rule == null) {
            return ApiResult.fail("发号规则不存在");
        }
        sysPlatformNumberRuleDtlRepoProc.delete(ids);

        // 对已存在的重新排序
        List<SysPlatformNumberRuleDtlDO> details = sysPlatformNumberRuleDtlRepoProc.queryDetails(ruleId);
        if (details.isEmpty()) {
            return ApiResult.ok(true);
        }
        var seq = 0;
        for (SysPlatformNumberRuleDtlDO detail : details) {
            detail.setSeq(seq);
            seq++;
        }
        sysNumberRuleDtlRepo.saveAll(details);

        // 同步给租户
        this.syncToTenant(rule.getAppCode(), rule.getRuleCode());
        return ApiResult.ok(true);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> removeByRuleId(Long ruleId) {
        if (ruleId == null) {
            return ApiResult.fail("规则ID为空");
        }

        var rule = numberRuleRepoProc.get(ruleId);
        if (rule == null) {
            return ApiResult.fail("发号规则不存在");
        }

        sysPlatformNumberRuleDtlRepoProc.deleteByRuleId(ruleId);

        // 同步给租户
        this.syncToTenant(rule.getAppCode(), rule.getRuleCode());

        return ApiResult.ok(ruleId);
    }

    @Override
    public ApiResult<List<SysPlatformNumberRuleDtlVO>> listByRuleId(Long ruleId) {
        if (ruleId == null) {
            return ApiResult.fail("规则ID为空");
        }

        List<SysPlatformNumberRuleDtlDO> details = sysPlatformNumberRuleDtlRepoProc.queryDetails(ruleId);
        if (details.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        Map<String, String> numberTypeUdc = getUdcNumberTypeMap();
        List<SysPlatformNumberRuleDtlVO> result = details.stream()
                .map(t -> {
                    SysPlatformNumberRuleDtlVO vo = CONVERT.doToVO(t);
                    vo.setNumberTypeName(numberTypeUdc.get(vo.getNumberType()));
                    return vo;
                }).collect(Collectors.toList());

        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<List<SysPlatformNumberRuleDtlDTO>> listDtoByRuleId(Long ruleId) {
        if (ruleId == null) {
            return ApiResult.fail("规则ID为空");
        }

        List<SysPlatformNumberRuleDtlDTO> result = sysPlatformNumberRuleDtlRepoProc.queryByRuleId(ruleId);
        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<SysPlatformNumberRuleDtlVO> oneById(Long id) {
        if (id == null) {
            return ApiResult.fail("ID为空");
        }

        SysPlatformNumberRuleDtlVO ruleDtlVO = sysPlatformNumberRuleDtlRepoProc.getOptional(id).map(CONVERT::doToVO).orElse(null);
        return ApiResult.ok(ruleDtlVO);
    }

    private Map<String, String> getUdcNumberTypeMap() {
        //        data: [{domainCode: "SYS", udcCode: "NUMBERING", udcVal: "RV", valDesc: "运行时动态值", es1: "", es2: null},…]
//        0: {domainCode: "SYS", udcCode: "NUMBERING", udcVal: "RV", valDesc: "运行时动态值", es1: "", es2: null}
//        1: {domainCode: "SYS", udcCode: "NUMBERING", udcVal: "FS", valDesc: "固定值", es1: "", es2: null}
//        2: {domainCode: "SYS", udcCode: "NUMBERING", udcVal: "DP", valDesc: "格式化日期时间", es1: "", es2: null}
//        3: {domainCode: "SYS", udcCode: "NUMBERING", udcVal: "NN", valDesc: "下一编号", es1: "", es2: null}
        Map<String, String> map = new HashMap<String, String>();
        map.put("RV", "运行时动态值");
        map.put("FS", "固定值");
        map.put("DP", "格式化日期时间");
        map.put("NN", "下一编号");
        return map;

        // return sysUdcService.getCodeMap("SYS", "NUMBERTYPE");
    }

    private void syncToTenant(String appCode, String ruleCode) {
        if (tenantProperties == null || Boolean.FALSE.equals(tenantProperties.isAutoSyncBasicData())) {
            return;
        }
        taskExecutor.execute(() -> {
            try {
                TimeUnit.SECONDS.sleep(2);
            } catch (InterruptedException ignored) {
            }
            tenantBasicDataService.syncSequence(appCode, ruleCode);
        });
    }

}
