package com.elitescloud.cloudt.platform.service.repo;

import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.service.common.constant.AppAuthTypeEnum;
import com.elitescloud.cloudt.system.service.model.entity.QSysPlatformAppDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysTenantAppDO;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformAppDO;
import com.elitescloud.cloudt.platform.model.params.app.QueryAppParam;
import com.querydsl.core.types.OrderSpecifier;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.JPAExpressions;
import org.springframework.stereotype.Repository;

import java.util.Collection;
import java.util.List;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/9/20
 */
@Repository
public class SysPlatformAppRepoProc extends BaseRepoProc<SysPlatformAppDO> {
    private static final QSysPlatformAppDO QDO = QSysPlatformAppDO.sysPlatformAppDO;
    private static final QSysTenantAppDO QDO_TENANT_APP = QSysTenantAppDO.sysTenantAppDO;

    public SysPlatformAppRepoProc() {
        super(QDO);
    }

    public List<SysPlatformAppDO> listByAppCodes(Collection<String> appCodes) {
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.appCode, appCodes)
                .build();
        return super.getList(predicate, defaultOrder());
    }

    /**
     * 所有启用状态的应用
     *
     * @return
     */
    public List<CodeNameParam> allEnabled() {
        var predicate = PredicateBuilder.builder()
                .andEq(true, QDO.appState, true)
                .build();
        return jpaQueryFactory.select(Projections.bean(CodeNameParam.class,
                        QDO.appCode.as("code"), QDO.appName.as("name")))
                .from(QDO)
                .where(predicate)
                .orderBy(defaultOrder())
                .fetch();
    }

    /**
     * 所有需要授权的的应用
     *
     * @return
     */
    public List<SysPlatformAppDO> allEnabled(Boolean outerApp, Boolean authed) {
        var predicate = this.predicateForOuterAuthed(outerApp, authed);
        return super.getList(predicate, defaultOrder());
    }

    /**
     * 分页查询管理
     *
     * @param param 查询参数
     * @return 应用列表
     */
    public PagingVO<SysPlatformAppDO> pageMng(QueryAppParam param) {
        var predicate = PredicateBuilder.builder()
                .andLike(QDO.appCode, param.getAppCode())
                .andLike(QDO.appName, param.getAppName())
                .build();
        return super.queryByPage(predicate, param.getPageRequest(), defaultOrder());
    }

    /**
     * 分页查询管理
     *
     * @param param 查询参数
     * @return 应用列表
     */
    public PagingVO<SysPlatformAppDO> pageQuery(QueryAppParam param) {
        var predicateAuth = this.predicateForOuterAuthed(param.getOuterApp(), param.getAuthed());

        var predicate = PredicateBuilder.builder()
                .andLike(QDO.appCode, param.getAppCode())
                .andLike(QDO.appName, param.getAppName())
                .andEq(QDO.appState, true)
                .and(predicateAuth)
                .build();
        return super.queryByPage(predicate, param.getPageRequest(), defaultOrder());
    }

    /**
     * 查询外部应用
     *
     * @param tenantId 租户ID
     * @return 应用列表
     */
    public List<SysPlatformAppDO> listOuterApp(Long tenantId) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.appState, true)
                .andEq(QDO.outerApp, true)
                .and(tenantId != null && tenantId.longValue() != TenantConstant.DEFAULT_TENANT_ID, () -> QDO.appCode.in(JPAExpressions.select(QDO_TENANT_APP.appCode).from(QDO_TENANT_APP).where(QDO_TENANT_APP.sysTenantId.eq(tenantId))))
                .build();
        return super.getList(predicate, defaultOrder());
    }

    /**
     * 所有需要授权的的应用
     *
     * @return
     */
    public List<SysPlatformAppDO> list(Boolean outerApp, Boolean authed) {
        var predicate = this.predicateForOuterAuthed(outerApp, authed);
        return jpaQueryFactory.selectFrom(QDO)
                .where(predicate)
                .orderBy(defaultOrder())
                .fetch();
    }

    private Predicate predicateForOuterAuthed(Boolean outerApp, Boolean authed) {
        return PredicateBuilder.builder()
                .andEq(true, QDO.appState, true)
                .andEq(outerApp != null, QDO.outerApp, outerApp)
                .and(authed != null, () -> {
                    if (authed == null) {
                        return null;
                    }

                    // 需要授权的
                    if (authed) {
                        if (outerApp == null) {
                            return QDO.outerApp.eq(false).or(QDO.authType.eq(AppAuthTypeEnum.AUTHED.name()));
                        }
                        return outerApp ? QDO.authType.eq(AppAuthTypeEnum.AUTHED.name()) : null;
                    }

                    // 不需要授权的
                    if (outerApp == null) {
                        return QDO.outerApp.eq(true).or(QDO.authType.eq(AppAuthTypeEnum.UNNECESSARY.name()));
                    }
                    if (outerApp) {
                        return QDO.authType.eq(AppAuthTypeEnum.UNNECESSARY.name());
                    }
                    return QDO.outerApp.eq(false).and(QDO.authType.eq(AppAuthTypeEnum.AUTHED.name()));
                })
                .build();
    }

    private OrderSpecifier<Integer> defaultOrder() {
        return QDO.appOrder.asc();
    }
}
