package com.elitescloud.cloudt.platform.provider.area;

import lombok.Getter;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

import javax.net.ssl.*;
import java.io.Serializable;
import java.security.SecureRandom;
import java.security.cert.X509Certificate;
import java.util.*;

/**
 * 行政区域编码抓取.
 * <p>
 * 数据来源：<a href="https://www.mca.gov.cn/n156/n2679/index.html">民政部</a>
 * <p>
 * 目前最新的是2025年公布的数据：<a href="https://www.mca.gov.cn/mzsj/xzqh/2025/202401xzqh.html">2024年中华人民共和国县以上行政区划代码</a>
 *
 * @author Kaiser（wang shao）
 * @date 2025/6/11 周三
 */
public class GrabAreaCodeTool {
    private static final Logger logger = LoggerFactory.getLogger(GrabAreaCodeTool.class);
    /**
     * 编码长度
     */
    private static final int CODE_LEN = 6;
    /**
     * 编码段的长度
     */
    private static final int BLOCK_LEN = 2;

    private final Map<String, AreaNode> allNodes;

    public GrabAreaCodeTool(String url) {
        this.allNodes = analyzeUrl(url);
    }

    public List<AreaNode> buildTree() {
        return buildTree(false);
    }

    public List<AreaNode> buildTree(boolean forceLevel) {
        if (allNodes.isEmpty()) {
            return Collections.emptyList();
        }

        List<AreaNode> rootNodes = new ArrayList<>();
        for (AreaNode value : new ArrayList<>(allNodes.values())) {
            if (value.level == 1) {
                // 省
                rootNodes.add(value);
                continue;
            }

            AreaNode parent = findRealParentNode(value, forceLevel);
            if (parent != null) {
                parent.children.add(value);
                value.actualParentCode = parent.code;
            }
        }
        return rootNodes;
    }

    private AreaNode findRealParentNode(AreaNode value, boolean createOnNotExist) {
        AreaNode parent = findParentNode(value);
        if (parent == null) {
            return null;
        }
        if (!createOnNotExist || parent.code.equals(value.parentCode)) {
            return parent;
        }

        AreaNode newParentNode = null;
        var cityCodeT = value.code.substring(BLOCK_LEN, BLOCK_LEN * 2);
        if (cityCodeT.equals("90")) {
            // 省直辖县
            newParentNode = new AreaNode(value.parentCode, "直辖县");
        } else {
            // 直辖市
            newParentNode = new AreaNode(value.parentCode, parent.name.replace("市", ""));
        }
        newParentNode.virtual = true;

        parent.children.add(newParentNode);
        allNodes.put(newParentNode.code, newParentNode);
        return newParentNode;
    }

    private AreaNode findParentNode(AreaNode value) {
        String codeDefault = "0".repeat(CODE_LEN);

        String parentCode = value.parentCode;
        int level = value.level - 1;
        while (true) {
            AreaNode parent = allNodes.get(parentCode);
            if (parent != null) {
                return parent;
            }

            level--;
            if (level == 0) {
                return null;
            }
            int index = level * BLOCK_LEN;
            parentCode = parentCode.substring(0, index) + codeDefault.substring(index);
        }
    }

    /**
     * 解析网页
     *
     * @param url 网页地址
     * @return 行政区划数据
     */
    private Map<String, AreaNode> analyzeUrl(String url) {
        // 设置ssl证书
        SSLSocketFactory originalSocketFactory = HttpsURLConnection.getDefaultSSLSocketFactory();
        HostnameVerifier hostnameVerifier = HttpsURLConnection.getDefaultHostnameVerifier();
        SSLContext sslContext = buildSSLContext();
        HttpsURLConnection.setDefaultSSLSocketFactory(sslContext.getSocketFactory());
        HttpsURLConnection.setDefaultHostnameVerifier((hostname, session) -> true);

        try {
            logger.info("开始抓取行政区域数据，数据地址：{}", url);
            // 抓取网页
            Document doc = Jsoup.connect(url).timeout(30000).get();

            // 提取表格数据
            Element table = doc.select("table").first();
            Assert.notNull(table, "获取行政区数据失败");

            Elements rows = table.select("tr");
            Map<String, AreaNode> allNodes = new LinkedHashMap<>();
            AreaNode areNode = null;
            String code, name;
            for (Element row : rows) {
                Elements cols = row.select("td");

                int startIndex = cols.size() >= 3 ? 1 : 0;
                code = cols.get(startIndex).text().trim();
                name = cols.get(startIndex + 1).text().trim();
                if (!code.isEmpty() && !name.isEmpty() && code.matches("\\d{6}")) {
                    areNode = new AreaNode(code, name);
                    allNodes.put(areNode.code, areNode);
                }
            }

            logger.info("解析完成，共获取 {} 个有效行政区划", allNodes.size());
            return allNodes;
        } catch (Exception e) {
            throw new RuntimeException("获取行政区数据异常", e);
        } finally {
            // 还原ssl证书
            if (originalSocketFactory != null) {
                HttpsURLConnection.setDefaultSSLSocketFactory(originalSocketFactory);
            }
            if (hostnameVerifier != null) {
                HttpsURLConnection.setDefaultHostnameVerifier(hostnameVerifier);
            }
        }
    }

    private static SSLContext buildSSLContext() {
        SSLContext sslContext = null;
        try {
            sslContext = SSLContext.getInstance("TLS");
            sslContext.init(null, new TrustManager[]{new X509TrustManager() {
                public void checkClientTrusted(X509Certificate[] chain, String authType) {
                }

                public void checkServerTrusted(X509Certificate[] chain, String authType) {
                }

                public X509Certificate[] getAcceptedIssuers() {
                    return new X509Certificate[0];
                }
            }}, new SecureRandom());
        } catch (Exception e) {
            throw new RuntimeException("构建SslContext异常", e);
        }

        return sslContext;
    }

    @Getter
    public static class AreaNode implements Serializable {
        private static final long serialVersionUID = 2653467613223198117L;

        /**
         * 节点编码
         */
        private final String code;

        /**
         * 节点名称
         */
        private final String name;

        /**
         * 节点层级
         * <p>
         * 1-省级、2-市级、3-县级
         */
        private final int level;

        /**
         * 上级节点编码
         */
        private final String parentCode;

        /**
         * 实际的父节点
         */
        private String actualParentCode;

        /**
         * 虚拟节点
         */
        private boolean virtual;

        /**
         * 下级节点
         */
        private final List<AreaNode> children = new ArrayList<>();

        public AreaNode(String nodeCode, String nodeName) {
            this.code = nodeCode;
            this.name = nodeName.replace("*", "");
            this.level = determineLevel(nodeCode);
            this.parentCode = determineParentCode(nodeCode, level);
        }

        @Override
        public String toString() {
            var label = getDirectLabel();
            return name + "(" + code + ")" + (label == null ? "" : " " + label);
        }

        public String getDirectLabel() {
            if (level == 1) {
                if (!children.isEmpty() && (children.get(0).level == 3 || children.get(0).virtual)) {
                    return "直辖市";
                }
                return null;
            }

            if (level == 3) {
                if (code.startsWith("90", 2)) {
                    return "省直辖县";
                }
                return null;
            }

            return null;
        }

        public void preview(String indent, StringBuilder content) {
            content.append(indent + this + " (层级: " + level + ") \n");
            if (children.isEmpty()) {
                return;
            }
            children.sort(Comparator.comparing(node -> node.code));
            for (AreaNode child : children) {
                child.preview(indent + "  ", content);
            }
        }

        private String determineParentCode(String code, int level) {
            // 省级
            if (level == 1) {
                return null;
            }

            String codeDefault = "0".repeat(CODE_LEN);

            int index = (level - 1) * BLOCK_LEN;
            return code.substring(0, index) + codeDefault.substring(index);
        }

        private int determineLevel(String code) {
            int blocks = CODE_LEN / BLOCK_LEN;
            String codeDefault = "0".repeat(CODE_LEN);

            for (int i = 1; i <= blocks; i++) {
                int index = i * BLOCK_LEN;
                String tempCode = code.substring(0, index) + codeDefault.substring(index);
                if (tempCode.equals(code)) {
                    return i;
                }
            }

            return blocks;
        }
    }
}
