package com.elitescloud.cloudt.platform.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.constant.CacheKey;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.platform.convert.AppConvert;
import com.elitescloud.cloudt.system.cacheable.SysCacheAppRpcService;
import com.elitescloud.cloudt.system.constant.SysTerminalType;
import com.elitescloud.cloudt.system.service.common.constant.AppAuthTypeEnum;
import com.elitescloud.cloudt.system.service.common.constant.PlatformAppTypeEnum;
import com.elitescloud.cloudt.platform.model.params.app.AddAppParam;
import com.elitescloud.cloudt.platform.model.params.app.QueryAppParam;
import com.elitescloud.cloudt.platform.model.params.app.UpdateAppParam;
import com.elitescloud.cloudt.platform.model.vo.SysPlatformAppVO;
import com.elitescloud.cloudt.platform.service.SysPlatformAppService;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformAppRepo;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformAppRepoProc;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformMenusRepo;
import com.elitescloud.cloudt.system.service.callback.AppChangedCallback;
import com.elitescloud.cloudt.system.vo.SysUdcVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * @Description:
 * @author: niu.chen
 * @date: 2022.09.15
 **/
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
@Slf4j
@RequiredArgsConstructor
public class SysPlatformAppServiceImpl implements SysPlatformAppService {

    private final SysPlatformAppRepo sysPlatformAppRepo;
    private final SysPlatformAppRepoProc sysPlatformAppRepoProc;
    private final RedisUtils redisUtils;
    private final ObjectProvider<AppChangedCallback> appChangedCallbacks;

    @Autowired
    private SysCacheAppRpcService cacheAppRpcService;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> addApp(AddAppParam addAppParam) {
        var list = sysPlatformAppRepo.findByAppCode(addAppParam.getAppCode());
        if (!list.isEmpty()) {
            return ApiResult.fail("AppCode重复");
        }

        addAppParam.setAppState(ObjectUtil.defaultIfNull(addAppParam.getAppState(), true));
        addAppParam.setAppOrder(ObjectUtil.defaultIfNull(addAppParam.getAppOrder(), 0));
        addAppParam.setOuterApp(ObjectUtil.defaultIfNull(addAppParam.getOuterApp(), false));
        addAppParam.setAuthType(ObjectUtil.defaultIfBlank(addAppParam.getAuthType(), AppAuthTypeEnum.AUTHED.name()));
        addAppParam.setAdaptedTerminal(ObjectUtil.defaultIfBlank(addAppParam.getAdaptedTerminal(), SysTerminalType.WEB.getValue()));

        var saveDo = AppConvert.INSTANCE.saveParamToDo(addAppParam);
        var id = sysPlatformAppRepo.save(saveDo).getId();

        // 清空缓存
        clearCache();

        // 回调
        for (AppChangedCallback callback : appChangedCallbacks) {
            callback.onSave(true, saveDo.getAppCode());
        }

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> deleteFlagApp(Long id) {
        sysPlatformAppRepo.findById(id).ifPresentOrElse(sysPlatformAppDO ->
                {
                    sysPlatformAppDO.setDeleteFlag(1);
                    sysPlatformAppRepo.save(sysPlatformAppDO);

                    // 清空缓存
                    clearCache();

                    // 回调
                    for (AppChangedCallback callback : appChangedCallbacks) {
                        callback.onDelete(sysPlatformAppDO.getAppCode(), false);
                    }
                }
                ,
                () -> {
                    throw new BusinessException("id不存在");
                });

        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> deleteApp(Long id) {
        var appDo = sysPlatformAppRepo.findById(id);
        if (appDo.isPresent()) {
            log.info("App应用删除：{}", appDo.get());
            sysPlatformAppRepo.deleteById(id);
            // 清空缓存
            clearCache();

            // 回调
            for (AppChangedCallback callback : appChangedCallbacks) {
                callback.onDelete(appDo.get().getAppCode(), true);
            }
        } else {
            throw new BusinessException("id不存在");
        }
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> updateApp(Long id, UpdateAppParam updateAppParam) {
        updateAppParam.setAppOrder(ObjectUtil.defaultIfNull(updateAppParam.getAppOrder(), 0));
        updateAppParam.setOuterApp(ObjectUtil.defaultIfNull(updateAppParam.getOuterApp(), false));
        updateAppParam.setAuthType(ObjectUtil.defaultIfBlank(updateAppParam.getAuthType(), AppAuthTypeEnum.AUTHED.name()));
        updateAppParam.setAdaptedTerminal(ObjectUtil.defaultIfBlank(updateAppParam.getAdaptedTerminal(), SysTerminalType.WEB.getValue()));

        AtomicReference<Boolean> allowUpdate = new AtomicReference<>(false);
        sysPlatformAppRepo.findById(id).ifPresentOrElse(sysPlatformAppDO -> {
//                    if (!sysPlatformAppDO.getAppCode().equals(updateAppParam.getAppCode())) {
//                        var list = sysPlatformAppRepo.findByAppCode(updateAppParam.getAppCode());
//                        if (!list.isEmpty()) {
//                            allowUpdate.set(false);
//                            return;
//                            // throw new BusinessException("Code已经存在重复:" + updateAppParam.getAppCode());
//                        }
//                    }
//                    sysPlatformAppDO.setAppCode(updateAppParam.getAppCode());
                    sysPlatformAppDO.setAppName(updateAppParam.getAppName());
                    sysPlatformAppDO.setAppType(updateAppParam.getAppType());
                    sysPlatformAppDO.setAdaptedTerminal(updateAppParam.getAdaptedTerminal());
                    sysPlatformAppDO.setAppDescribe(updateAppParam.getAppDescribe());
                    sysPlatformAppDO.setAppOrder(updateAppParam.getAppOrder());
                    sysPlatformAppDO.setOuterApp(updateAppParam.getOuterApp());
                    sysPlatformAppDO.setAuthType(updateAppParam.getAuthType());
                    sysPlatformAppDO.setUrl(updateAppParam.getUrl());
                    sysPlatformAppDO.setTokenUrl(updateAppParam.getTokenUrl());
                    sysPlatformAppDO.setIcon(updateAppParam.getIcon());
                    sysPlatformAppRepo.save(sysPlatformAppDO);
                    allowUpdate.set(true);

                    // 清空缓存
                    clearCache();

                    // 回调
                    for (AppChangedCallback callback : appChangedCallbacks) {
                        callback.onSave(false, sysPlatformAppDO.getAppCode());
                    }
                },
                () -> {
                    throw new BusinessException("id不存在");
                });
        if (allowUpdate.get()) {
            return ApiResult.ok(true);
        } else {
            return ApiResult.fail("编码已存在");
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> updateAppState(Long id, Boolean state) {
        sysPlatformAppRepo.findById(id).ifPresentOrElse(sysPlatformAppDO ->
                {
                    sysPlatformAppDO.setAppState(state);
                    sysPlatformAppRepo.save(sysPlatformAppDO);

                    // 清空缓存
                    clearCache();

                    // 回调
                    for (AppChangedCallback callback : appChangedCallbacks) {
                        callback.onEnabled(sysPlatformAppDO.getAppCode(), Boolean.TRUE.equals(state));
                    }
                },

                () -> {
                    throw new BusinessException("id不存在");
                });
        return ApiResult.ok();
    }

    @Override
    public ApiResult<SysPlatformAppVO> getApp(Long id) {
        var appDO = sysPlatformAppRepo.findById(id);
        if (appDO.isPresent()) {
            return ApiResult.ok(AppConvert.INSTANCE.sysPlatformAppDOToSysPlatformAppVO(appDO.get()));
        } else {
            return ApiResult.fail("id不存在");
        }
    }

    @Override
    public ApiResult<PagingVO<SysPlatformAppVO>> pageMng(QueryAppParam queryAppParam) {
        var pagingVo
                = sysPlatformAppRepoProc.pageMng(queryAppParam)
                .map(AppConvert.INSTANCE::sysPlatformAppDOToSysPlatformAppVO);
        return ApiResult.ok(pagingVo);
    }

    @Override
    public ApiResult<List<SysUdcVO>> getAppType() {

        return ApiResult.ok(
                List.of(PlatformAppTypeEnum.APP_TYPE_SYS.getUdcVO()
                        , PlatformAppTypeEnum.APP_TYPE_BUS.getUdcVO()
                )
        );
    }

    @Override
    public ApiResult<List<CodeNameParam>> all() {
        var result = sysPlatformAppRepoProc.allEnabled();
        try {
            redisUtils.set(CacheKey.CACHE_APP_ALL, result.stream().map(t -> new CodeNameParam(t.getCode(), t.getName())).collect(Collectors.toList()));
        } catch (Exception e) {
            log.error("缓存APP异常", e);
        }

        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<PagingVO<SysPlatformAppVO>> queryByPage(QueryAppParam queryAppParam) {
        var pagingVo
                = sysPlatformAppRepoProc.pageQuery(queryAppParam)
                .map(AppConvert.INSTANCE::sysPlatformAppDOToSysPlatformAppVO);
        return ApiResult.ok(pagingVo);
    }

    @Override
    public ApiResult<List<SysPlatformAppVO>> listApp(Boolean outerApp, Boolean authed) {
        var voList = sysPlatformAppRepoProc.list(outerApp, authed).stream()
                .map(AppConvert.INSTANCE::sysPlatformAppDOToSysPlatformAppVO)
                .collect(Collectors.toList());
        return ApiResult.ok(voList);
    }

    private void clearCache() {
        try {
            redisUtils.del(CacheKey.CACHE_APP_ALL);
        } catch (Exception e) {
            log.error("清空app缓存异常", e);
        }

        cacheAppRpcService.clearCache();
    }
}
