package com.elitescloud.cloudt.platform.service.impl;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.platform.convert.AppConvert;
import com.elitescloud.cloudt.platform.convert.MenusConvert;
import com.elitescloud.cloudt.platform.model.constant.PlatformAdminTypeEnum;
import com.elitescloud.cloudt.system.constant.PlatformAppMenusTypeEnum;
import com.elitescloud.cloudt.system.constant.PlatformMenusNodeEnum;
import com.elitescloud.cloudt.platform.model.entity.SysPlatformAdminMenusDO;
import com.elitescloud.cloudt.platform.model.params.admin.AddAdminMenusParam;
import com.elitescloud.cloudt.platform.model.vo.GetAdminMenusVO;
import com.elitescloud.cloudt.platform.model.vo.SysPlatformAppVO;
import com.elitescloud.cloudt.platform.model.vo.SysPlatformMenusVO;
import com.elitescloud.cloudt.platform.service.SysPlatformAdminMenusService;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformAdminMenusRepoProc;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformAppRepoProc;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformMenusRepoProc;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @Description: 管理员菜单
 * @author: niu.chen
 * @date: 2022.09.15
 **/
@Service
@Slf4j
@RequiredArgsConstructor
public class SysPlatformAdminMenusServiceImpl implements SysPlatformAdminMenusService {
    private final SysPlatformAdminMenusRepoProc adminMenusRepoProc;
    private final SysPlatformMenusRepoProc menusRepoProc;
    private final SysPlatformAppRepoProc appRepoProc;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Boolean> addAdminMenus(AddAdminMenusParam addParams) {
        Assert.notNull(addParams.getAdminType(), "管理员类型为空");
        var adminType = addParams.getAdminType().name();

        List<SysPlatformAdminMenusDO> adminMenus = new ArrayList<>(128);
        if (CollUtil.isNotEmpty(addParams.getMenusCodeList())) {
            List<PlatformAppMenusTypeEnum> menuTypes = getMenuTypes(addParams.getAdminType());

            // 获取菜单类型的
            var allCodes = menusRepoProc.getCodes(null, menuTypes, PlatformMenusNodeEnum.MENUS);
            if (!allCodes.isEmpty()) {
                addParams.getMenusCodeList().stream()
                        .filter(allCodes::contains)
                        .map(t -> {
                            var adminMenu = new SysPlatformAdminMenusDO();
                            adminMenu.setMenusCode(t);
                            adminMenu.setAdminType(adminType);
                            return adminMenu;
                        })
                        .forEach(adminMenus::add);
            }
        }

        adminMenusRepoProc.deleteByAdminType(adminType);
        if (!adminMenus.isEmpty()) {
            adminMenusRepoProc.save(adminMenus);
        }
        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<GetAdminMenusVO> getAdminMenus(PlatformAdminTypeEnum adminType) {
        Assert.notNull(adminType, "管理员类型为空");

        List<SysPlatformAppVO> sysPlatformAppVOS = new ArrayList<>(0);

        List<SysPlatformMenusVO> sysPlatformMenusVOS = adminMenusRepoProc.queryMenusByAdminType(adminType).stream()
                //菜单状态是启用的
                .filter(t -> Boolean.TRUE.equals(t.getMenusState()))
                .map(MenusConvert.INSTANCE::sysPlatformDOToSysPlatformVO)
                .collect(Collectors.toList());

        Set<String> appCodeList = sysPlatformMenusVOS.isEmpty() ? Collections.emptySet() :
                sysPlatformMenusVOS.stream().map(SysPlatformMenusVO::getMenusAppCode)
                        .collect(Collectors.toSet());

        if (!appCodeList.isEmpty()) {
            sysPlatformAppVOS = appRepoProc.listByAppCodes(appCodeList)
                    .stream()
                    .filter(t -> Boolean.TRUE.equals(t.getAppState()))
                    .map(AppConvert.INSTANCE::sysPlatformAppDOToSysPlatformAppVO)
                    .collect(Collectors.toList());
        }

        GetAdminMenusVO getAdminMenusVO = new GetAdminMenusVO();
        getAdminMenusVO.setAdminType(adminType.name());
        getAdminMenusVO.setAdminTypeName(adminType.getDescription());
        getAdminMenusVO.setSysPlatformAppVOS(sysPlatformAppVOS);
        getAdminMenusVO.setSysPlatformMenusVOS(sysPlatformMenusVOS);
        return ApiResult.ok(getAdminMenusVO);
    }

    private List<PlatformAppMenusTypeEnum> getMenuTypes(PlatformAdminTypeEnum adminType) {
        switch (adminType) {
            case SYS_ADMIN:
                return List.of(PlatformAppMenusTypeEnum.MENUS_TYPE_PLATFORM);
            case TENANT_ADMIN:
            case TENANT_ORG_ADMIN:
                return List.of(PlatformAppMenusTypeEnum.MENUS_TYPE_SYS);
            default:
                throw new BusinessException("暂不支持的管理员类型");
        }
    }
}
