package com.elitescloud.cloudt.system.service;

import com.elitescloud.cloudt.Application;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.param.SysUdcQueryParam;
import com.elitescloud.cloudt.system.vo.SysUdcComboVO;
import com.elitescloud.cloudt.system.vo.SysUdcVO;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Map;

/**
 * UDC服务接口
 *
 * @author Michael Li
 * @date 2020/6/25
 */
@FeignClient(name = Application.NAME, path = SysUdcService.URI)
@Validated
public interface SysUdcService {
    String URI = Application.URI_PREFIX + "/udc";

    /**
     * 检索分类码
     *
     * @param param 分类查询、分页和排序对象
     * @return 分页结果集
     */
    @PostMapping(value = "/search")
    PagingVO<SysUdcVO> search(SysUdcQueryParam param);

    /**
     * 根据关键字段，返回对应的分类码
     *
     * @param domainCode 领域码
     * @param udcCode    分类码
     * @param udcVal     分类码值
     * @return 分类码对象
     */
    @GetMapping(value = "/findOneSysUdc")
    SysUdcVO findOneSysUdc(@NotBlank(message = "领域码为空") @RequestParam(value = "domainCode") String domainCode,
                           @NotBlank(message = "UDC编码为空") @RequestParam(value = "udcCode") String udcCode,
                           @NotBlank(message = "UDC值为空") @RequestParam(value = "udcVal") String udcVal);

    /**
     * 根据关键字段，返回对应的分类码
     *
     * @param domain  领域码
     * @param udc     分类码
     * @param valDesc
     * @return 分类码的值和值描述
     */
    @GetMapping(value = "/listCodeCombosValLike")
    List<SysUdcComboVO> listCodeCombosValLike(@NotBlank(message = "领域码为空") @RequestParam(value = "domain") String domain,
                                              @NotBlank(message = "UDC编码为空") @RequestParam(value = "udc") String udc,
                                              @RequestParam(value = "valDesc", required = false) String valDesc);

    /**
     * 根据关键字段，返回对应的分类码
     * 模糊查询itemGroup1 的值和code
     * 蒙华
     *
     * @param domain  领域码
     * @param udcCode
     * @param udc     以udc参数开头的
     * @return 模糊查询
     */
    @GetMapping(value = "/listSubCodeLike")
    List<SysUdcComboVO> listSubCodeLike(@NotBlank(message = "领域码为空") @RequestParam(value = "domain") String domain,
                                        @NotBlank(message = "UDC编码为空") @RequestParam(value = "udcCode") String udcCode,
                                        @RequestParam(value = "udc", required = false) String udc);

    /**
     * 根据关键字段，返回对应的分类码
     * 我模糊查询itemGroup1 的值和code
     * 蒙华
     *
     * @param domain 领域码
     * @param
     * @return 模糊查询
     */
    @GetMapping(value = "/listCodeLike")
    List<SysUdcComboVO> listCodeLike(@NotBlank(message = "领域码为空") @RequestParam(value = "domain") String domain,
                                     @NotBlank(message = "UDC编码为空") @RequestParam(value = "udcCode") String udcCode,
                                     @RequestParam(value = "udc", required = false) String udc);

    /**
     * 根据ID返回对应记录
     *
     * @param id 记录ID
     * @return 记录对象
     */
    @GetMapping(value = "/findOneById")
    SysUdcVO findOneById(@NotNull(message = "ID为空") @RequestParam(value = "id") Long id);

    /**
     * 获取用于下来选择的分类码
     *
     * @param domain 领域码
     * @param udc    分类码
     * @return 分类码的值和值描述
     */
    @GetMapping(value = "/listCodeCombos")
    List<SysUdcComboVO> listCodeCombos(@NotBlank(message = "领域码为空") @RequestParam(value = "domain") String domain,
                                       @NotBlank(message = "UDC编码为空") @RequestParam(value = "udc") String udc);

    /**
     * 获取用于下来选择的分类码,udc的AOP翻译功能使用
     *
     * @param domain 领域码
     * @param udc    分类码
     * @return 分类码的值和值描述
     */
    @GetMapping(value = "/listCodeCombosUdc")
    List<SysUdcComboVO> listCodeCombosUdc(@NotBlank(message = "领域码为空") @RequestParam(value = "domain") String domain,
                                          @NotBlank(message = "UDC编码为空") @RequestParam(value = "udc") String udc);

    /**
     * 获取分类码的Map对应列表
     *
     * @param domain 领域码
     * @param udc    分类码
     * @return 分类码的值和描述
     */
    @GetMapping(value = "/getCodeMap")
    Map<String, String> getCodeMap(@NotBlank(message = "领域码为空") @RequestParam(value = "domain") String domain,
                                   @NotBlank(message = "UDC编码为空") @RequestParam(value = "udc") String udc);


    /**
     * 获取全部分类码的Map对应列表
     *
     * @return 分类码的值和描述
     */
    @GetMapping(value = "/getCodeMapAll")
    Map<String, String> getCodeMapAll();

    /**
     * 根据关键字段，返回对应的分类码
     * 我模糊查询itemGroup1 的值和code
     * 蒙华
     *
     * @param udc udc值列表
     * @return UDC下拉列表
     */
    @PostMapping(value = "/findByUdcValues")
    List<SysUdcComboVO> findByUdcValues(@RequestBody List<String> udc);


    @PostMapping(value = "/findUdcByIds")
    List<SysUdcVO> findUdcByIds(@RequestBody List<Long> ids);

    /**
     * 组合UDC key 用于map 远程调用查询 udcDesc描述使用.
     * 参数:
     * 1 domainCode 领域code,
     * 2 udcCode udc编码
     * 3 valValue udc值
     * <p>
     * 组合采用&分割
     **/
    static String combinationUdcKey(String domainCode, String udcCode, String valValue) {
        String link = "&";
        return domainCode + link +
                udcCode + link +
                valValue;
    }
}
