package com.elitescloud.cloudt.system.provider.org;

import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.swagger.feignAPI.DocumentedFeignApi;
import com.elitescloud.cloudt.Application;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.dto.SysOrgBasicDTO;
import com.elitescloud.cloudt.system.dto.req.OrgUpsertDTO;
import com.elitescloud.cloudt.system.dto.req.SysOrgPageQueryDTO;
import com.elitescloud.cloudt.system.dto.resp.SysOrgDetailRespDTO;
import com.elitescloud.cloudt.system.dto.resp.SysOrgPageRespDTO;
import com.elitescloud.cloudt.system.param.SysOrgQueryDTO;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 组织服务接口.
 *
 * @author Kaiser（wang shao）
 * @date 2/23/2023
 */
@FeignClient(name = Application.NAME, path = OrgRpcService.URI)
@Validated
@DocumentedFeignApi( description = "组织服务")
public interface OrgRpcService {
    String URI = Application.URI_PREFIX + "/org";

    /**
     * 保存组织
     * <p>
     * 如果id不为空，则是修改
     *
     * @param upsertDTO 组织信息
     * @return 组织ID
     */
    @PostMapping(value = "/upsert")
    @DocumentedFeignApi( description = "保存组织 如果id不为空，则是修改")
    ApiResult<Long> upsert(@RequestBody @Valid OrgUpsertDTO upsertDTO);

    /**
     * 修改启用状态
     *
     * @param id      组织ID
     * @param enabled 启用状态
     * @return 组织ID
     */
    @PatchMapping(value = "/updateEnabled")
    @DocumentedFeignApi( description = "修改启用状态")
    ApiResult<Long> updateEnabled(@NotNull(message = "组织ID为空") @RequestParam("id") Long id,
                                  @NotNull(message = "启用状态为空") @RequestParam("enabled") Boolean enabled);

    /**
     * 删除组织
     * <p>
     * 如果下面有子组织，则不可直接删除
     *
     * @param id 组织ID
     * @return 组织ID
     */
    @DeleteMapping(value = "/delete")
    @DocumentedFeignApi( description = "删除组织 如果下面有子组织，则不可直接删除")
    ApiResult<Long> delete(@NotNull(message = "组织ID为空") @RequestParam("id") Long id);

    /**
     * 根据ID获取组织信息
     *
     * @param id 组织ID
     * @return 组织信息
     */
    @GetMapping(value = "/getById")
    @DocumentedFeignApi( description = "根据ID获取组织信息")
    ApiResult<SysOrgBasicDTO> getById(@NotNull(message = "组织ID为空") @RequestParam("id") Long id);

    /**
     * 根据ID获取组织详细信息
     *
     * @param id             组织ID
     * @param withParentType 指定类型的上级组织，{@link com.elitescloud.cloudt.system.constant.OrgType}
     * @return 组织信息
     */
    @GetMapping(value = "/getDetailById")
    @DocumentedFeignApi( description = "根据ID获取组织详细信息")
    ApiResult<SysOrgDetailRespDTO> getDetailById(@NotNull(message = "组织ID为空") @RequestParam("id") Long id,
                                                 @RequestParam(value = "withParentType", required = false) String withParentType);

    /**
     * 根据编码获取组织信息
     *
     * @param code 组织编码
     * @return 组织信息
     */
    @GetMapping(value = "/getByCode")
    @DocumentedFeignApi( description = "根据编码获取组织信息")
    ApiResult<SysOrgBasicDTO> getByCode(@NotBlank(message = "组织编码为空") @RequestParam("code") String code);

    /**
     * 根据编码获取组织详细信息
     *
     * @param code           组织编码
     * @param withParentType 指定类型的上级组织，{@link com.elitescloud.cloudt.system.constant.OrgType}
     * @return 组织信息
     */
    @GetMapping(value = "/getDetailByCode")
    @DocumentedFeignApi( description = "根据编码获取组织详细信息")
    ApiResult<SysOrgDetailRespDTO> getDetailByCode(@NotBlank(message = "组织编码为空") @RequestParam("code") String code,
                                                   @RequestParam(value = "withParentType", required = false) String withParentType);

    /**
     * 根据编码获取上级组织信息
     *
     * @param code       组织编码
     * @param parentType 指定上级类型
     * @return 组织信息
     */
    @GetMapping(value = "/getParentByCode")
    @DocumentedFeignApi( description = "根据编码获取上级组织信息")
    ApiResult<SysOrgBasicDTO> getParentByCode(@NotBlank(message = "组织编码为空") @RequestParam("code") String code,
                                              @RequestParam(value = "parentType", required = false) String parentType
    );

    /**
     * 根据编码获取上级组织信息
     *
     * @param codes      组织编码
     * @param parentType 指定上级类型
     * @return 组织信息
     */
    @PostMapping(value = "/getParentsByCode")
    @DocumentedFeignApi( description = "根据编码获取上级组织信息")
    ApiResult<Map<String, SysOrgBasicDTO>> getParentByCode(@NotEmpty(message = "组织编码为空") @RequestBody Set<String> codes,
                                                           @RequestParam(value = "parentType", required = false) String parentType
    );

    /**
     * 查询组织列表
     *
     * @param queryDTO 查询参数
     * @return 组织列表
     */
    @PostMapping(value = "/queryList")
    @DocumentedFeignApi( description = "查询组织列表")
    ApiResult<List<SysOrgBasicDTO>> queryList(@RequestBody SysOrgQueryDTO queryDTO);

    /**
     * 分页查询组织
     *
     * @param queryDTO 查询参数
     * @return 组织列表
     */
    @PostMapping(value = "/queryPage")
    @DocumentedFeignApi( description = "分页查询组织")
    ApiResult<PagingVO<SysOrgPageRespDTO>> queryPage(@RequestBody SysOrgPageQueryDTO queryDTO);

    /**
     * 根据ID查询组织编码和名称
     *
     * @param ids 组织ID
     * @return 组织信息
     */
    @PostMapping(value = "/queryCodeAndName")
    @DocumentedFeignApi( description = "根据ID查询组织编码和名称")
    ApiResult<List<IdCodeNameParam>> queryCodeAndName(@RequestBody Set<Long> ids);

    /**
     * 根据ID查询组织编码和名称
     *
     * @param id 组织ID
     * @return 组织信息
     */
    @GetMapping(value = "/getCodeAndName")
    @DocumentedFeignApi( description = "根据ID查询组织编码和名称")
    ApiResult<IdCodeNameParam> getCodeAndName(@NotNull(message = "组织ID为空") @RequestParam("id") Long id);

    /**
     * 根据ID查询组织编码和名称
     *
     * @param codes 组织编码
     * @return 组织信息
     */
    @PostMapping(value = "/queryIdAndName")
    @DocumentedFeignApi( description = "根据ID查询组织编码和名称")
    ApiResult<List<IdCodeNameParam>> queryIdAndName(@RequestBody Set<String> codes);

    /**
     * 根据编码查询组织编码和名称
     *
     * @param code 组织编码0
     * @return 组织信息
     */
    @GetMapping(value = "/getIdAndName")
    @DocumentedFeignApi( description = "根据编码查询组织编码和名称")
    ApiResult<IdCodeNameParam> getIdAndName(@NotBlank(message = "组织编码为空") @RequestParam("code") String code);
}
