package com.elitescloud.cloudt.system.datasource.service;

import com.elitescloud.boot.swagger.feignapi.DocumentedFeignApi;
import com.elitescloud.cloudt.Application;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.datasource.*;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * @author : chen.niu
 * @description :
 * @date : 2024-04-29 17:01
 */
@FeignClient(name = Application.NAME, path = SysPlatformDatabaseSourceRpcService.URI)
@Validated
@DocumentedFeignApi( description = "数据库源服务")
public interface SysPlatformDatabaseSourceRpcService {
    String URI = Application.URI_PREFIX + "/databaseSource";

    /**
     * 查询平台数据库源信息。
     *
     * @param queryParam 查询参数对象，包含过滤条件等。
     * @return 返回数据库源信息的列表，包含查询结果状态和数据。
     */
    @PostMapping(value = "/query")
    @DocumentedFeignApi( description = "查询平台数据库源信息")
    ApiResult<List<SysPlatformDatabaseSourceVO>> query(@RequestBody @Valid SysPlatformDatabaseSourceQueryParam queryParam);


    /**
     * 查询平台数据库源密码。
     */
    @GetMapping(value = "/queryPasswordById")
    @DocumentedFeignApi( description = "查询平台数据库源密码")
    ApiResult<String> queryPasswordById(@NotNull(message = "记录ID为空") @RequestParam("id") Long id);

    /**
     * 根据ID获取数据库源信息。
     *
     * @param id 数据库源的ID，不能为空。
     * @return 返回特定数据库源的信息，包含查询结果状态和数据。
     */
    @GetMapping(value = "/getById")
    @DocumentedFeignApi( description = "根据ID获取数据库源信息")
    ApiResult<SysPlatformDatabaseSourceVO> getById(@NotNull(message = "记录ID为空") @RequestParam Long id);

    /**
     * 根据数据库源代码获取信息。
     *
     * @param dbsCode 数据库源的代码，不能为空。
     * @return 返回匹配的数据库源信息，包含查询结果状态和数据。
     */
    @GetMapping(value = "/getByDbsCode")
    @DocumentedFeignApi( description = "根据数据库源代码获取信息")
    ApiResult<SysPlatformDatabaseSourceVO> getByDbsCode(
            @NotBlank(message = "记录ID为空") @RequestParam String dbsCode);

    /**
     * 保存新的数据库源信息。
     *
     * @param saveParam 包含要保存的数据库源信息的参数对象。
     * @return 返回保存成功的数据库源ID，包含操作结果状态和数据。
     */
    @PostMapping(value = "/save")
    @DocumentedFeignApi( description = "保存新的数据库源信息")
    ApiResult<Long> save(@RequestBody @Valid SysPlatformDatabaseSourceSaveParam saveParam);

    /**
     * 更新现有的数据库源信息。
     *
     * @param id      要更新的数据库源ID。
     * @param upParam 包含更新内容的参数对象。
     * @return 返回更新操作影响的行数，包含操作结果状态和数据。
     */
    @PostMapping(value = "/update")
    @DocumentedFeignApi( description = "更新现有的数据库源信息")
    ApiResult<Long> update(
            @RequestParam @NotNull(message = "id不能空") Long id,
            @RequestBody @Valid SysPlatformDatabaseSourceUpParam upParam);

    /**
     * 根据ID删除数据库源信息。
     *
     * @param id 要删除的数据库源ID，不能为空。
     * @return 返回删除操作影响的行数，包含操作结果状态和数据。
     */
    @DocumentedFeignApi( description = "根据ID删除数据库源信息")
    @PostMapping(value = "/delete")
    ApiResult<Long> delete(@RequestParam Long id);

    /**
     * 测试数据库连接配置的可用性。
     *
     * @param config 包含数据库连接配置信息的对象。
     * @return 返回连接测试的结果，通常为成功或失败的消息，包含操作结果状态和数据。
     */
    @PostMapping(value = "/testConnection")
    @DocumentedFeignApi( description = "测试数据库连接配置的可用性")
    ApiResult<String> testConnection(@RequestBody @Valid DatabaseConfigDTO config);

    @PostMapping(value = "/testConnectionById")
    @DocumentedFeignApi( description = "测试数据库连接配置的可用性，根据ID")
    ApiResult<String> testConnectionById(@RequestParam Long dbDataSourceId);
}
