package com.elitescloud.cloudt.system.quartz.service.provider;

import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.util.DatetimeUtil;
import com.elitescloud.boot.util.ExceptionsUtil;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.cloudt.system.quartz.common.SysQuartzConstant;
import com.elitescloud.cloudt.system.quartz.common.SysQuartzJobFiredType;
import com.elitescloud.cloudt.system.quartz.model.entity.SysQuartzTriggerRecordDO;
import com.elitescloud.cloudt.system.quartz.service.repo.SysQuartzJobRepoProc;
import com.elitescloud.cloudt.system.quartz.service.repo.SysQuartzTriggerRecordRepoProc;
import org.quartz.Job;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.support.TransactionTemplate;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Map;

/**
 * 统一任务代理.
 *
 * @author Kaiser（wang shao）
 * @date 2025/9/21 周日
 */
public class SysQuartzJobDelegate implements Job {
    private static final Logger logger = LoggerFactory.getLogger(SysQuartzJobDelegate.class);

    @Override
    public void execute(JobExecutionContext context) throws JobExecutionException {
        String jobCode = context.getJobDetail().getKey().getName();
        logger.info("任务调度开始：{}", jobCode);

        // 初始化任务记录
        var recordId = initTriggerRecord(context);

        // 获取任务信息
        var jobInfo = getSysQuartzJobRepoProc().getByJobCode(jobCode);
        if (jobInfo == null) {
            logger.error("任务不存在: {}", jobCode);
            getSysQuartzTriggerRecordRepoProc().updateFailed(recordId, "任务不存在", null);
            return;
        }
        if (Boolean.FALSE.equals(jobInfo.getEnabled())) {
            logger.error("任务已禁用：{}", jobCode);
            getSysQuartzTriggerRecordRepoProc().updateFailed(recordId, "任务已禁用", null);
            return;
        }

        // 加载任务执行器
        Job job = null;
        try {
            job = getJob(jobInfo.getJobClass());
        } catch (Exception e) {
            logger.error("任务执行器加载失败：{}, {}", jobCode, jobInfo.getJobClass(), e);
            getSysQuartzTriggerRecordRepoProc().updateFailed(recordId, e.getMessage(), ExceptionsUtil.stackTraceAllToString(e));
            return;
        }

        // 执行任务
        try {
            job.execute(context);
            getSysQuartzTriggerRecordRepoProc().updateSuccess(recordId);
        } catch (JobExecutionException e) {
            logger.error("任务执行器执行异常：{}, {}", jobCode, jobInfo.getJobClass(), e);
            getSysQuartzTriggerRecordRepoProc().updateFailed(recordId, e.getMessage(), ExceptionsUtil.stackTraceAllToString(e));
            return;
        }
    }

    private SysQuartzJobFiredType obtainFiredType(JobExecutionContext context) {
        var mergedJobData = context.getMergedJobDataMap();

        // 运行一次
        var runOnce = mergedJobData.getBooleanValue(SysQuartzConstant.JOB_DATA_KEY_SOURCE_RUNONCE);
        if (runOnce) {
            return SysQuartzJobFiredType.MANUAL_RUNONCE;
        }

        // 恢复运行
        if (context.isRecovering()) {
            return SysQuartzJobFiredType.QUARTZ_RECOVERY;
        }

        // 默认任务调度
        return SysQuartzJobFiredType.QUARTZ;
    }

    private Long initTriggerRecord(JobExecutionContext context) {
        SysQuartzTriggerRecordDO recordDO = new SysQuartzTriggerRecordDO();
        recordDO.setJobCode(context.getJobDetail().getKey().getName());
        recordDO.setFireTime(DatetimeUtil.date2LocalDateTime(context.getFireTime()));
        recordDO.setScheduledTime(DatetimeUtil.date2LocalDateTime(context.getScheduledFireTime()));
        recordDO.setFiredType(obtainFiredType(context).name());
//        recordDO.setFinishedTime();
        recordDO.setSuccess(false);
//        recordDO.setFailMsg();
//        recordDO.setFullFailMsg();
        recordDO.setJobDataMap(JSONUtil.toJsonString(context.getJobDetail().getJobDataMap()));
        recordDO.setTriggerJobDataMap(JSONUtil.toJsonString(context.getTrigger().getJobDataMap()));

        getSysQuartzTriggerRecordRepoProc().save(recordDO);
        return recordDO.getId();
    }

    private SysQuartzJobRepoProc getSysQuartzJobRepoProc() {
        if (sysQuartzJobRepoProc == null) {
            sysQuartzJobRepoProc = SpringContextHolder.getBean(SysQuartzJobRepoProc.class);
        }
        return sysQuartzJobRepoProc;
    }

    private SysQuartzTriggerRecordRepoProc getSysQuartzTriggerRecordRepoProc() {
        if (sysQuartzTriggerRecordRepoProc == null) {
            sysQuartzTriggerRecordRepoProc = SpringContextHolder.getBean(SysQuartzTriggerRecordRepoProc.class);
        }
        return sysQuartzTriggerRecordRepoProc;
    }

    private TransactionTemplate getTransactionTemplate() {
        if (transactionTemplate == null) {
            transactionTemplate = SpringContextHolder.getBean(TransactionTemplate.class);
        }
        return transactionTemplate;
    }

    public Job getJob(String jobClassName) {
        var job = jobInstanceMap.get(jobClassName);
        if (job != null) {
            return job;
        }

        Constructor<? extends Job> constructor = null;
        try {
            Class<? extends Job> jobClass = (Class<? extends Job>) Class.forName(jobClassName);
            constructor = jobClass.getConstructor();
        } catch (ClassNotFoundException e) {
            throw new IllegalArgumentException("任务类不存在：" + jobClassName);
        } catch (NoSuchMethodException e) {
            throw new IllegalArgumentException("任务类不存在无参构造方法：" + jobClassName);
        }
        try {
            job = constructor.newInstance();
            logger.info("任务类实例化成功：{}", jobClassName);
        } catch (InstantiationException | IllegalAccessException | InvocationTargetException e) {
            throw new IllegalArgumentException("任务类实例化异常：" + jobClassName);
        }
        jobInstanceMap.put(jobClassName, job);

        return job;
    }

    private TransactionTemplate transactionTemplate;
    private SysQuartzJobRepoProc sysQuartzJobRepoProc;
    private SysQuartzTriggerRecordRepoProc sysQuartzTriggerRecordRepoProc;
    private final Map<String, Job> jobInstanceMap = new HashMap<>(64);
}
