package com.elitescloud.cloudt.system.seq.common;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.LockUtil;
import com.elitescloud.cloudt.system.constant.SysNumType;
import com.elitescloud.cloudt.system.seq.model.bo.SysSeqRuleDtlBO;
import com.elitescloud.cloudt.system.seq.service.NextNumGenerateService;
import com.elitescloud.cloudt.system.seq.service.SeqNumGenerateService;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StopWatch;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * 发号器工厂.
 *
 * @author Kaiser（wang shao）
 * 2022/11/29
 */
@Component
@Log4j2
public class SeqNumFactory {
    private static final Map<String, DateTimeFormatter> FORMATTER_MAP = new HashMap<>();
    private static final String PLACEHOLDER_NEXT_NUMBER = "##{NN}##";

    @Autowired
    private NextNumGenerateService generateService;
    @Autowired
    private SeqNumGenerateService supportService;

    /**
     * 根据发号规则ID，生成样例号
     *
     * @param ruleId        发号规则ID
     * @param runtimeValues 基于UDC选择值的列表
     * @return 生成样例号，下一编号用默认，1
     */
    public String generateSampleCode(@NotNull Long ruleId, List<String> runtimeValues) {
        Assert.notNull(ruleId, "发号规则ID为空");

        // 生成编号
        var values = executeGenerateCode(ruleId, d -> List.of(1L), runtimeValues, 1);
        if (CollUtil.isNotEmpty(values)) {
            return values.get(0);
        }
        return null;
    }

    /**
     * 根据发号规则编码，生成样例号
     *
     * @param appCode       应用编码
     * @param ruleCode      发号规则编码
     * @param runtimeValues 基于UDC选择值的列表
     * @return 生成样例号，洗衣编号默认用，1
     */
    public String generateSampleCode(@NotBlank String appCode, @NotBlank String ruleCode, List<String> runtimeValues) {
        Assert.hasText(appCode, "应用编码为空");
        Assert.hasText(ruleCode, "规则编码为空");

        Long ruleId = supportService.getRuleIdByCode(appCode, ruleCode);
        Assert.notNull(ruleId, "发号规则不存在");

        return this.generateSampleCode(ruleId, runtimeValues);
    }

    /**
     * 根据发号规则ID，生成对应号码
     *
     * @param ruleId        发号规则ID
     * @param runtimeValues 选择的UDC的值列表
     * @return 生成的号码
     */
    public String generateCode(@NotNull Long ruleId, List<String> runtimeValues) {
        var values = this.generateCode(ruleId, runtimeValues, 1);
        if (CollUtil.isNotEmpty(values)) {
            return values.get(0);
        }
        return null;
    }

    /**
     * 根据发号规则ID，生成对应号码
     *
     * @param ruleId        发号规则ID
     * @param runtimeValues 选择的UDC的值列表
     * @param num           发号数量
     * @return 生成的号码
     */
    public List<String> generateCode(@NotNull Long ruleId, List<String> runtimeValues, int num) {
        Assert.notNull(ruleId, "发号规则ID为空");

        // 生成编号
        StopWatch watch = new StopWatch();
        watch.start();
        var code = executeGenerateCode(ruleId, d -> {
                    try {
                        return nextValue(d.getAppCode(), d.getNumberPattern(), d.getNnLen(), num);
                    } catch (Exception e) {
                        log.error("生成下一编号失败：{}", e.getMessage());
                        throw new BusinessException("生成下一编号失败", e);
                    }
                }
                , runtimeValues, num);
        watch.stop();
        log.info("发号完成，耗时{}ms", watch.getTotalTimeMillis());
        return code;
    }

    /**
     * 根据发号规则编码，生成对应号码
     *
     * @param appCode       应用编码
     * @param ruleCode      发号规则编码
     * @param runtimeValues 选择UDC的值列表
     * @return 生成的号码
     */
    public String generateCode(@NotBlank String appCode, @NotBlank String ruleCode, List<String> runtimeValues) {
        var values = this.generateCode(appCode, ruleCode, runtimeValues, 1);
        if (CollUtil.isNotEmpty(values)) {
            return values.get(0);
        }
        return null;
    }

    /**
     * 根据发号规则编码，生成对应号码
     *
     * @param appCode       应用编码
     * @param ruleCode      发号规则编码
     * @param runtimeValues 选择UDC的值列表
     * @param num           发号数量
     * @return 生成的号码
     */
    public List<String> generateCode(@NotBlank String appCode, @NotBlank String ruleCode, List<String> runtimeValues, int num) {
        Assert.hasText(appCode, "应用编码为空");
        Assert.hasText(ruleCode, "规则编码为空");

        Long ruleId = supportService.getRuleIdByCode(appCode, ruleCode);
        if (ruleId == null) {
            throw new BusinessException("发号规则不存在");
        }
        return this.generateCode(ruleId, runtimeValues, num);
    }

    /**
     * 下一序号
     *
     * @param nnId  下一序号的ID
     * @param nnLen 下一序号的长度
     * @return 下一序号
     */
    public Long nextValue(@NotNull Long nnId, Integer nnLen) {
        var values = this.nextValue(nnId, nnLen, 1);
        if (CollUtil.isNotEmpty(values)) {
            return values.get(0);
        }
        return null;
    }

    /**
     * 下一序号
     *
     * @param nnId  下一序号的ID
     * @param nnLen 下一序号的长度
     * @param num   生成下一序号的数量
     * @return 下一序号
     */
    public List<Long> nextValue(@NotNull Long nnId, Integer nnLen, int num) {
        Assert.notNull(nnId, "下一序号的ID为空");
        Assert.isTrue(num > 0, "下一序号的数量不能小于1");

        // 保证只有一个创建
        var lockKey = "nv:" + nnId;
        List<Long> valueList = null;
        StopWatch stopWatch = new StopWatch();
        stopWatch.start();
        try {
            valueList = executeByLock(() -> generateService.nextNumber(nnId, nnLen, num), lockKey);
        } catch (InterruptedException e) {
            stopWatch.stop();
            log.error("{}生成下一序号异常：{}，耗时{}ms", nnId, e.getMessage(), stopWatch.getTotalTimeMillis());
            throw new BusinessException("生成下一序号异常", e);
        }
        stopWatch.stop();
        log.info("{}生成下一序号结束，耗时：{}ms", nnId, stopWatch.getTotalTimeMillis());
        if (CollectionUtils.isEmpty(valueList)) {
            throw new BusinessException("生成下一序号失败");
        }
        return valueList;
    }

    /**
     * 下一序号
     *
     * @param appCode 应用编码
     * @param nnCode  下一序号的编号
     * @return 下一序号
     */
    public Long nextValue(@NotBlank String appCode, @NotBlank String nnCode, Integer nnlen) {
        return this.nextValue(appCode, nnCode, nnlen, false);
    }

    /**
     * 下一序号
     *
     * @param appCode 应用编码
     * @param nnCode  下一序号的编号
     * @param num     生成下一序号的数量
     * @return 下一序号
     */
    public List<Long> nextValue(@NotBlank String appCode, @NotBlank String nnCode, Integer nnlen, int num) {
        return this.nextValue(appCode, nnCode, nnlen, false, num);
    }

    /**
     * 下一序号
     *
     * @param nnCode            下一序号的编号
     * @param nnlen             下一序号的宽度
     * @param createOnNotExists 如果不存在是否自动初始化创建
     * @return 下一序号
     */
    public Long nextValue(@NotBlank String appCode, @NotBlank String nnCode, Integer nnlen, boolean createOnNotExists) {
        var values = this.nextValue(appCode, nnCode, nnlen, createOnNotExists, 1);
        if (CollUtil.isNotEmpty(values)) {
            return values.get(0);
        }
        return null;
    }

    /**
     * 下一序号
     *
     * @param nnCode            下一序号的编号
     * @param nnlen             下一序号的宽度
     * @param createOnNotExists 如果不存在是否自动初始化创建
     * @param num               创建数量
     * @return 下一序号
     */
    public List<Long> nextValue(@NotBlank String appCode, @NotBlank String nnCode, Integer nnlen, boolean createOnNotExists, int num) {
        Assert.hasText(appCode, "应用编码为空");
        Assert.hasText(nnCode, "下一序号的编码为空");

        // 获取下一序号的ID
        Long id = supportService.getNextNumberIdByCode(appCode, nnCode);
        if (id != null) {
            return this.nextValue(id, nnlen, num);
        }

        // 不存在则创建
        Assert.isTrue(createOnNotExists, "未找到" + nnCode + "下一序号");
        id = createNextValue(appCode, nnCode);

        // 生成下一序号
        return this.nextValue(id, nnlen, num);
    }

    private Long createNextValue(String appCode, String nnCode) {
        // 保证只有一个创建
        var lockKey = CharSequenceUtil.blankToDefault(appCode, "def") + ":nn:" + nnCode;
        try {
            return executeByLock(() -> {
                Long id = supportService.getNextNumberIdByCode(appCode, nnCode);
                if (id != null) {
                    return id;
                }

                // 没有则创建
                id = generateService.createNextNumberForInit(appCode, nnCode);
                if (id == null) {
                    id = supportService.getNextNumberIdByCode(appCode, nnCode);
                }
                return id;
            }, lockKey);
        } catch (InterruptedException e) {
            log.error("创建下一序号异常：{}", e.getMessage());
            throw new BusinessException("创建下一序号异常", e);
        }
    }

    private List<String> executeGenerateCode(@NonNull Long ruleId, Function<SysSeqRuleDtlBO, List<Long>> nextNumberFunction,
                                             List<String> runtimeValues, int num) {
        Assert.notNull(ruleId, "发号器规则ID为空");
        Assert.isTrue(num > 0, "发号数量不能小于1");

        // 查询发号规则
        var ruleDetails = supportService.getRuleDetailsByRuleId(ruleId);
        Assert.isTrue(!ruleDetails.isEmpty(), "未查询到发号规则配置");

        // 生成编号
        return this.executeGenerateCode(ruleId, ruleDetails, nextNumberFunction, runtimeValues, num);
    }

    private List<String> executeGenerateCode(@NonNull Long ruleId, @NotEmpty List<SysSeqRuleDtlBO> ruleDetails,
                                             Function<SysSeqRuleDtlBO, List<Long>> nextNumberFunction,
                                             List<String> runtimeValues, int num) {
        StringBuilder code = new StringBuilder();
        SysSeqRuleDtlBO lastNextNumberRuleDtl = null;
        if (num > 1) {
            lastNextNumberRuleDtl = this.obtainLastNextNumberRuleDtl(ruleDetails);
            Assert.notNull(lastNextNumberRuleDtl, "批量发号失败，未发现递增序号配置");
        }

        int offset = 0;
        for (SysSeqRuleDtlBO ruleDetail : ruleDetails) {
            // 取号类型
            Assert.state(StringUtils.hasText(ruleDetail.getNumberType()), "发号器规则【" + ruleId + "】配置有误，取号类型为空");

            // 根据不同取号类型生成号码
            LocalDateTime now = LocalDateTime.now();
            String pattern = ruleDetail.getNumberPattern();
            var numType = SysNumType.parse(ruleDetail.getNumberType());
            Assert.notNull(numType, "未知规则类型：" + ruleDetail.getNumberType());
            switch (numType) {
                case FS: {
                    // 固定值
                    if (!StringUtils.hasText(pattern)) {
                        throw new BusinessException("发号器规则【" + ruleId + "】配置有误，固定值的模式为空");
                    }
                    code.append(pattern);
                    break;
                }
                case DP: {
                    // 日期格式
                    if (!StringUtils.hasText(pattern)) {
                        throw new BusinessException("发号器规则【" + ruleId + "】配置有误，日期格式的模式为空");
                    }
                    code.append(FORMATTER_MAP.computeIfAbsent(pattern, DateTimeFormatter::ofPattern).format(now));
                    break;
                }
                case NN: {
                    // 自增
                    Integer len = ruleDetail.getNnLen();
                    if (len == null) {
                        throw new BusinessException("发号器规则【" + ruleId + "】配置有误，自增序号宽度为空");
                    }
                    if (lastNextNumberRuleDtl != null && lastNextNumberRuleDtl.getId().longValue() == ruleDetail.getId()) {
                        // 临时用占位符
                        code.append(PLACEHOLDER_NEXT_NUMBER);
                    } else {
                        var nextNumbers = nextNumberFunction.apply(ruleDetail);
                        if (CollUtil.isEmpty(nextNumbers)) {
                            throw new BusinessException("生成下一编号失败");
                        }
                        code.append(preZero(nextNumbers.get(0), len));
                    }
                    break;
                }
                default:
                    if (runtimeValues != null && offset < runtimeValues.size()) {
                        if (CharSequenceUtil.equals("NN", pattern)) {
                            Integer len = ruleDetail.getNnLen();
                            if (len == null) {
                                throw new BusinessException("发号器规则【" + ruleId + "】配置有误，自增序号宽度为空");
                            }
                            String nnCode = runtimeValues.get(offset);
                            Long nextNumber = null;
                            try {
                                nextNumber = nextValue(ruleDetail.getAppCode(), nnCode, len, true);
                            } catch (Exception e) {
                                log.error("生成下一编号失败：{}", e.getMessage());
                                throw new BusinessException("生成下一编号失败", e);
                            }
                            code.append(preZero(nextNumber, len));
                        } else {
                            code.append(runtimeValues.get(offset));
                        }
                        offset++;
                    }
            }
        }

        if (lastNextNumberRuleDtl == null) {
            // 非批量的
            return List.of(code.toString());
        }
        // 批量的
        return this.produceCodeBatch(code.toString(), lastNextNumberRuleDtl, nextNumberFunction);
    }

    private List<String> produceCodeBatch(String code, SysSeqRuleDtlBO ruleDtl, Function<SysSeqRuleDtlBO, List<Long>> nextNumberFunction) {
        var nextNumberList = nextNumberFunction.apply(ruleDtl);
        Assert.notEmpty(nextNumberList, "生成下一编号失败");

        List<String> codeList = new ArrayList<>();
        for (Long nextNumber : nextNumberList) {
            codeList.add(code.replace(PLACEHOLDER_NEXT_NUMBER, preZero(nextNumber, ruleDtl.getNnLen())));
        }
        return codeList;
    }

    private SysSeqRuleDtlBO obtainLastNextNumberRuleDtl(List<SysSeqRuleDtlBO> ruleDetails) {
        for (int i = ruleDetails.size() - 1; i >= 0; i--) {
            var dtl = ruleDetails.get(i);
            if (SysNumType.NN.name().equals(dtl.getNumberType())) {
                return dtl;
            }
        }
        return null;
    }

    private String preZero(long val, int len) {
        Assert.isTrue(len > 0, "宽度必须大于0");

        String format = "%0" + len + "d";
        return String.format(format, val);
    }

    private <T> T executeByLock(Supplier<T> supplier, String lockName) throws InterruptedException {
        return LockUtil.executeByLock(lockName, supplier, Duration.ofMinutes(30), "当前请求量过大，请稍后再试");
    }
}
