package com.elitescloud.cloudt.system.service.model.entity;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.common.annotation.BusinessParam;
import com.elitescloud.boot.constant.Gender;
import com.elitescloud.cloudt.common.annotation.Comment;
import com.elitescloud.boot.model.entity.BaseModel;
import com.elitescloud.cloudt.system.constant.UserSourceType;
import lombok.Getter;
import lombok.Setter;
import lombok.experimental.Accessors;
import org.hibernate.annotations.DynamicInsert;
import org.hibernate.annotations.DynamicUpdate;
import org.springframework.util.StringUtils;

import javax.persistence.*;
import java.time.LocalDate;
import java.time.LocalDateTime;

/**
 * 系统用户账号
 *
 * @author Michael Li
 * 2020-06-15
 */
@Entity
@Table(name = "sys_user", indexes = {
        @Index(name = "idx_sys_user_username", columnList = "username", unique = true),
        @Index(name = "idx_sys_user_outerkey", columnList = "outerKey"),
        @Index(name = "idx_sys_user_mobile", columnList = "mobile"),
        @Index(name = "idx_sys_user_email", columnList = "email"),
        @Index(name = "idx_sys_user_source", columnList = "sourceType"),
        @Index(name = "idx_sys_user_cas_user", columnList = "casUserId"),
})
@Comment("系统用户账号")
@DynamicUpdate
@DynamicInsert
@Getter
@Setter
@Accessors(chain = true)
public class SysUserDO extends BaseModel {
    private static final long serialVersionUID = -7840804475330315230L;

    @Comment("登录账号")
    @Column(nullable = false)
    @BusinessParam(dataPermissionEnabled = false, fieldPermissionEnabled = true)
    private String username;

    @Comment(value = "外部系统唯一标识", defaultValue = "''")
    @Column()
    private String outerKey;

    @Comment("登录密码")
    @Column(nullable = false)
    private String password;

    @Comment(value = "密码是否需要重新设置", defaultValue = "1")
    @Column()
    private Boolean needReset;

    @Comment("姓名-姓氏")
    @Column()
    private String lastName;

    @Comment("姓名-名字")
    @Column()
    private String firstName;

    /**
     * {@link com.elitescloud.boot.constant.Gender}
     */
    @Column()
    @Comment("性别，（male-男，female-女，secret-保密）")
    private String gender;

    @Column()
    @Comment("昵称")
    private String nickName;

    @Column()
    @Comment("生日")
    private LocalDate birthDate;

    @Comment("手机号")
    @Column()
    private String mobile;

    @Comment("邮箱")
    @Column()
    private String email;

    @Comment(value = "是否启用", defaultValue = "1")
    @Column()
    @BusinessParam(dataPermissionEnabled = true, fieldPermissionEnabled = true)
    private Boolean enabled;

    @Comment("上次登录时间")
    @Column()
    private LocalDateTime lastLoginTime;

    @Comment("密码过期时间")
    @Column()
    private LocalDateTime passwordExpiredTime;

    @Comment("账号过期时间")
    @Column()
    private LocalDateTime expiredTime;

    @Column()
    @Comment("身份证号")
    private String idCard;

    @Column()
    @Comment("国家编码")
    private String countryCode;

    @Column()
    @Comment("省编码")
    private String provinceCode;

    @Column()
    @Comment("市编码")
    private String cityCode;

    @Column()
    @Comment("县编码")
    private String countyCode;

    @Column()
    @Comment("详细地址")
    private String address;

    @Column()
    @Comment("个性签名")
    private String personalSign;

    /**
     * 账号来源
     * {@link UserSourceType}
     */
    @Comment("账号来源, sys： 系统创建；wec：微信创建；oth：其他方式")
    @Column()
    private String sourceType;

    @Comment("用户头像地址，可直接访问，如微信头像地址等")
    @Lob
    private String avatarUrl;

    @Comment("用户头像在文件服务器中的标识")
    @Column()
    private String avatarCode;

    @Comment(value = "是否需要向统一身份认证中心同步", defaultValue = "1")
    @Column()
    private Boolean syncCas;

    @Comment("在统一身份认证中心的ID")
    @Column()
    private Long casUserId;

    /**
     * 获取全名
     *
     * @return 全名
     */
    public String getFullName() {
        if (!StringUtils.hasText(lastName)) {
            return firstName;
        }

        return lastName + CharSequenceUtil.blankToDefault(firstName, "");
    }

    /**
     * 获取性别
     *
     * @return 性别
     */
    public String getGenderName() {
        if (CharSequenceUtil.isBlank(getGender())) {
            return null;
        }
        return new Gender(getGender()).getDescription();
    }

    @Override
    public int hashCode() {
        return getId().hashCode();
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof SysUserDO) {
            SysUserDO the = (SysUserDO) obj;
            if (getId() == null) {
                return the.getId() == null;
            }
            return getId().equals(the.getId());
        }
        return false;
    }
}
