package com.elitesland.cbpl.bpmn.controller;

import com.elitesland.cbpl.bpmn.service.TaskVersionService;
import com.elitesland.cbpl.bpmn.vo.param.TaskVersionPagingParamVO;
import com.elitesland.cbpl.bpmn.vo.param.TaskVersionQueryParamVO;
import com.elitesland.cbpl.bpmn.vo.param.TaskVersionSaveParamVO;
import com.elitesland.cbpl.bpmn.vo.resp.TaskVersionDetailVO;
import com.elitesland.cbpl.bpmn.vo.resp.TaskVersionPagingVO;
import com.elitesland.cbpl.bpmn.vo.resp.TaskVersionRespVO;
import com.elitesland.cbpl.tool.api.domain.HttpResult;
import com.elitesland.cbpl.tool.db.PagingVO;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * @author eric.hao
 * @since 2024/05/09
 */
@Slf4j
@Api(tags = "任务流程 - 发布版本表")
@RestController
@RequestMapping("/bpmn/version")
@RequiredArgsConstructor
public class TaskVersionController {

    private final TaskVersionService taskVersionService;

    @ApiOperation(value = "分页查询")
    @PostMapping(value = "/list")
    public HttpResult<PagingVO<TaskVersionPagingVO>> list(@RequestBody @Valid TaskVersionPagingParamVO query) {
        logger.info("[TaskVersion] query paging param = {}", query);
        return HttpResult.ok(taskVersionService.taskVersionPageBy(query));
    }

    @ApiOperation(value = "不分页查询")
    @PostMapping(value = "/query")
    public HttpResult<List<TaskVersionRespVO>> query(@RequestBody @Valid TaskVersionQueryParamVO query) {
        logger.info("[TaskVersion] query list param = {}", query);
        return HttpResult.ok(taskVersionService.taskVersionByParam(query));
    }

    @ApiOperation(value = "明细查询")
    @GetMapping(value = "/detail/{id}")
    public HttpResult<TaskVersionDetailVO> detail(@PathVariable("id") Long id) {
        logger.info("[TaskVersion] query detail by id = {}", id);
        return HttpResult.ok(taskVersionService.taskVersionById(id));
    }

    @ApiOperation(value = "新增或修改")
    @PostMapping(value = "/save")
    public HttpResult<Long> save(@RequestBody @Valid TaskVersionSaveParamVO saveParam) {
        logger.info("[TaskVersion] create/update record, saveParam = {}", saveParam);
        return HttpResult.ok(taskVersionService.save(saveParam));
    }

    @ApiOperation(value = "发布生效")
    @PatchMapping(value = "/{id}/v")
    public HttpResult<Void> published(@PathVariable("id") Long id) {
        logger.info("[TaskVersion] active by id = {}", id);
        taskVersionService.published(id);
        return HttpResult.ok();
    }

    @ApiOperation(value = "批量逻辑删除")
    @DeleteMapping(value = "/{id}/delete")
    public HttpResult<Long> delete(@PathVariable("id") Long id) {
        logger.info("[TaskVersion] physical delete by id = {}", id);
        return HttpResult.ok(taskVersionService.delete(id));
    }
}
