package com.elitesland.cbpl.bpmn.service;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.bpmn.constant.BpmnConstant;
import com.elitesland.cbpl.bpmn.spi.BpmnContextSpi;
import com.elitesland.cbpl.tool.core.exceptions.PhoenixException;
import com.yomahub.liteflow.builder.el.LiteFlowChainELBuilder;
import com.yomahub.liteflow.core.FlowExecutor;
import com.yomahub.liteflow.flow.FlowBus;
import com.yomahub.liteflow.flow.LiteflowResponse;
import com.yomahub.liteflow.flow.element.Chain;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.elitesland.cbpl.bpmn.config.BpmnProperties.BPMN_ENABLED;
import static com.elitesland.cbpl.bpmn.util.BpmnUtil.chainKey;

/**
 * @author eric.hao
 * @since 2024/05/08
 */
@Slf4j
@RequiredArgsConstructor
public class BpmnExecutor {

    private final FlowExecutor flowExecutor;
    private final InstanceExecutor instanceExecutor;
    @Autowired(required = false)
    private BpmnContextSpi bpmnContextSpi;

    private FlowExecutor getInstance() {
        if (!BPMN_ENABLED) {
            throw PhoenixException.unexpected("bpmn executor not enabled.");
        }
        return flowExecutor;
    }

    /**
     * 注册编排流程
     *
     * @param taskCode 流程任务编码
     * @param version  版本号
     * @param flowSpEL 流程表达式
     */
    public void register(String taskCode, Integer version, String flowSpEL) {
        LiteFlowChainELBuilder.createChain().setChainId(chainKey(taskCode, version)).setEL(flowSpEL).build();
    }

    /**
     * 销毁指定流程
     *
     * @param taskCode 流程任务编码
     * @param version  版本号
     */
    public void destroy(String taskCode, Integer version) {
        // 如果流程存在
        if (containChain(taskCode, version)) {
            FlowBus.removeChain(chainKey(taskCode, version));
        }
    }

    /**
     * 重新注册
     *
     * @param taskCode 流程任务编码
     * @param version  版本号
     * @param flowSpEL 流程表达式
     */
    public void refresh(String taskCode, Integer version, String flowSpEL) {
        destroy(taskCode, version);
        register(taskCode, version, flowSpEL);
    }

    /**
     * 执行流程
     *
     * @param taskCode 流程任务编码
     * @param version  版本号
     * @return 流程结果
     */
    public LiteflowResponse start(String taskCode, Integer version) {
        return start(taskCode, version, null);
    }

    /**
     * 执行流程
     *
     * @param taskCode 流程任务编码
     * @param version  版本号
     * @param param    请求参数
     * @return 流程结果
     */
    public LiteflowResponse start(String taskCode, Integer version, Object param) {
        // 内置支持的参数对象
        List<Class<?>> context = BpmnConstant.bpmnCommonContext();
        // 自定义扩展支持的参数对象
        if (ObjectUtil.isNotNull(bpmnContextSpi)) {
            List<Class<?>> extra = bpmnContextSpi.extraContext();
            if (CollUtil.isNotEmpty(extra)) {
                context.addAll(extra);
            }
        }
        return start(taskCode, version, param, context.toArray(new Class[0]));
    }

    /**
     * 执行流程
     *
     * @param taskCode              流程任务编码
     * @param version               版本号
     * @param param                 请求参数
     * @param contextBeanClazzArray 自定义上下文
     * @return 流程结果
     */
    public LiteflowResponse start(String taskCode, Integer version, Object param, Class<?>... contextBeanClazzArray) {
        var instance = instanceExecutor.instanceLog(taskCode, version, "手动执行");
        return instanceExecutor.record(instance, () -> getInstance().execute2Resp(chainKey(taskCode, version), param, contextBeanClazzArray));
    }

    /**
     * 流程是否存在
     *
     * @param taskCode 流程任务编码
     * @param version  版本号
     * @return 是否存在
     */
    public boolean containChain(String taskCode, Integer version) {
        return FlowBus.containChain(chainKey(taskCode, version));
    }

    /**
     * 所有已注册的流程ID
     */
    public Set<String> chainAll() {
        Map<String, Chain> chainMap = FlowBus.getChainMap();
        return chainMap.keySet();
    }
}
