package com.elitesland.cbpl.bpmn.service;

import com.elitesland.cbpl.bpmn.convert.TaskInstanceConvert;
import com.elitesland.cbpl.bpmn.spi.BpmnCompleteProvider;
import com.elitesland.cbpl.bpmn.util.BpmnUtil;
import com.elitesland.cbpl.bpmn.vo.param.TaskInstanceSaveParamVO;
import com.elitesland.cbpl.scheduling.domain.InstanceStatus;
import com.elitesland.cbpl.tool.core.exceptions.ExceptionUtils;
import com.yomahub.liteflow.flow.LiteflowResponse;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.function.Supplier;

/**
 * @author eric.hao
 * @since 2024/05/11
 */
@Slf4j
@RequiredArgsConstructor
public class InstanceExecutor {

    @Resource
    private TaskInstanceService taskInstanceService;
    @Autowired(required = false)
    private BpmnCompleteProvider bpmnCompleteProvider;

    /**
     * 实例初始化
     *
     * @param taskCode 流程任务编码
     * @param version  版本号
     * @param runType  执行方式
     * @return 保存参数
     */
    public TaskInstanceSaveParamVO instanceLog(String taskCode, Integer version, String runType) {
        TaskInstanceSaveParamVO saveParam = new TaskInstanceSaveParamVO();
        saveParam.setTaskCode(taskCode);
        saveParam.setVersion(version);
        saveParam.setStartTime(LocalDateTime.now());
        saveParam.setInstanceStatus(InstanceStatus.RUNNING.getCode());
        saveParam.setRemark(runType);
        Long instanceId = taskInstanceService.save(saveParam);
        saveParam.setId(instanceId);
        return saveParam;
    }

    /**
     * 记录实例日志
     *
     * @param instance 实例
     * @param runnable 执行内容
     */
    public LiteflowResponse record(TaskInstanceSaveParamVO instance, Supplier<LiteflowResponse> supplier) {
        LiteflowResponse result = BpmnUtil.errorResponse();
        // 1. 实例执行前的回调方法
        try {
            if (bpmnCompleteProvider != null) {
                var response = TaskInstanceConvert.INSTANCE.saveParamToVO(instance);
                bpmnCompleteProvider.start(response);
            }
        } catch (Exception e) {
            logger.error("[PHOENIX-BPMN] execute complete start error.", e);
        }

        // 2. 实例执行「目前仅支持同步执行」
        Throwable throwable = null;
        try {
            // 初始状态，按实例正常执行成功处理
            instance.setInstanceStatus(InstanceStatus.COMPLETE.getCode());
            result = supplier.get();
        } catch (Exception e) {
            throwable = e;
            instance.setInstanceStatus(InstanceStatus.INTERRUPT.getCode());
            instance.setErrorMessage(ExceptionUtils.formatException(e, 255));
            logger.error("[PHOENIX-BPMN] execute error: {}", ExceptionUtils.formatException(e));
        }

        // 3. 记录实例执行结果
        instance.setEndTime(LocalDateTime.now());
        taskInstanceService.update(instance);

        // 4. 实例执行完后的回调方法
        try {
            if (bpmnCompleteProvider != null) {
                var response = TaskInstanceConvert.INSTANCE.saveParamToVO(instance);
                bpmnCompleteProvider.whenComplete(response, throwable);
            }
        } catch (Exception e) {
            logger.error("[PHOENIX-BPMN] execute complete callback error.", e);
        }
        return result;
    }
}
