package com.elitesland.cbpl.bpmn.controller;

import com.elitesland.cbpl.bpmn.data.service.TaskDefineService;
import com.elitesland.cbpl.bpmn.data.service.TaskInstanceService;
import com.elitesland.cbpl.bpmn.data.vo.param.TaskDefinePagingParamVO;
import com.elitesland.cbpl.bpmn.data.vo.param.TaskDefineQueryParamVO;
import com.elitesland.cbpl.bpmn.data.vo.param.TaskDefineSaveParamVO;
import com.elitesland.cbpl.bpmn.data.vo.param.TaskInstanceDeleteParamVO;
import com.elitesland.cbpl.bpmn.data.vo.resp.TaskDefineDetailVO;
import com.elitesland.cbpl.bpmn.data.vo.resp.TaskDefinePagingVO;
import com.elitesland.cbpl.bpmn.data.vo.resp.TaskDefineRespVO;
import com.elitesland.cbpl.tool.api.domain.HttpResult;
import com.elitesland.cbpl.tool.db.PagingVO;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * @author eric.hao
 * @since 2024/05/09
 */
@Slf4j
@Api(tags = "任务流程 - 定义表")
@RestController
@RequestMapping("/bpmn/define")
@RequiredArgsConstructor
public class TaskDefineController {

    private final TaskDefineService taskDefineService;
    private final TaskInstanceService taskInstanceService;

    @ApiOperation(value = "分页查询")
    @PostMapping(value = "/list")
    public HttpResult<PagingVO<TaskDefinePagingVO>> list(@RequestBody TaskDefinePagingParamVO query) {
        logger.info("[TaskDefine] query paging param = {}", query);
        return HttpResult.ok(taskDefineService.taskDefinePageBy(query));
    }

    @ApiOperation(value = "不分页查询")
    @PostMapping(value = "/query")
    public HttpResult<List<TaskDefineRespVO>> query(@RequestBody TaskDefineQueryParamVO query) {
        logger.info("[TaskDefine] query list param = {}", query);
        return HttpResult.ok(taskDefineService.taskDefineByParam(query));
    }

    @ApiOperation(value = "明细查询")
    @GetMapping(value = "/detail/{id}")
    public HttpResult<TaskDefineDetailVO> detail(@PathVariable("id") Long id) {
        logger.info("[TaskDefine] query detail by id = {}", id);
        return HttpResult.ok(taskDefineService.taskDefineById(id));
    }

    @ApiOperation(value = "新增或修改")
    @PostMapping(value = "/save")
    public HttpResult<Long> save(@RequestBody @Valid TaskDefineSaveParamVO saveParam) {
        logger.info("[TaskDefine] create/update record, saveParam = {}", saveParam);
        return HttpResult.ok(taskDefineService.save(saveParam));
    }

    @ApiOperation(value = "启用")
    @PatchMapping(value = "/{id}/v")
    public HttpResult<Long> active(@PathVariable("id") Long id) {
        logger.info("[TaskDefine] active by id = {}", id);
        return HttpResult.ok(taskDefineService.updateStatusV(id));
    }

    @ApiOperation(value = "禁用")
    @PatchMapping(value = "/{id}/x")
    public HttpResult<Long> inactive(@PathVariable("id") Long id) {
        logger.info("[TaskDefine] inactive by id = {}", id);
        return HttpResult.ok(taskDefineService.updateStatusX(id));
    }

    @ApiOperation(value = "批量逻辑删除")
    @PutMapping(value = "/delete")
    public HttpResult<Void> updateDeleteFlag(@RequestBody List<Long> ids) {
        logger.info("[TaskDefine] batch logic delete by ids = {}", ids);
        taskDefineService.updateDeleteFlag(ids);
        return HttpResult.ok();
    }

    @ApiOperation(value = "日志删除策略-CRON表达式预览")
    @GetMapping(value = "/strategy/preview")
    public HttpResult<List<String>> strategyPreview() {
        return HttpResult.ok(taskInstanceService.strategyPreview());
    }

    @ApiOperation(value = "删除实例日志")
    @DeleteMapping(value = "/delete/instance")
    public HttpResult<Long> delete(@Valid @RequestBody TaskInstanceDeleteParamVO paramVO) {
        logger.info("[PHOENIX-BPMN] delete instance param = {}", paramVO);
        return HttpResult.ok(taskInstanceService.deleteInstance(paramVO, "INVOKE"));
    }
}
